<?php

namespace App\ToolsClass;

/** 正则 */
class RegexTool
{
    // /u 表示按unicode(utf-8)匹配（主要针对多字节比如汉字）
    // /i 表示不区分大小写（如果表达式里面有 a， 那么 A 也是匹配对象）
    // /s 表示将字符串视为单行来匹配
    // \d匹配一个数字字符。等价于 [0-9]。
    // \w匹配包括下划线的任何单词字符。等价于’[A-Za-z0-9_]’。
    // \x{4e00}-\x{9fa5} 匹配中文
    // {6,16} 匹配字符个数。
    // preg_match_all("|{{([^^]*?).DATA}}|u", $query, $matches); 匹配字符中间的内容

    /*********************** 匹配数字 ******************************** */


    /**
     * 整数 （整数）
     *
     * @param string $num
     * @return boolean
     */
    public static function ints($num)
    {
        $rule = "/^-?\d+$/";
        return preg_match($rule, $num);
    }


    /**
     * 正整数 （正整数）
     *
     * @param string $num
     * @return boolean
     */
    public static function int($num)
    {
        $rule = "/^[0-9]*[1-9][0-9]*$/";
        return preg_match($rule, $num);
    }

    /**
     * 非负整数（正整数 + 0）
     *
     * @param string $num
     * @return boolean
     */
    public static function intz($num)
    {
        $rule = "/^\d+$/";
        return preg_match($rule, $num);
    }


    /**
     * 负整数 （负整数）
     *
     * @param string $num
     * @return boolean
     */
    public static function _int($num)
    {
        $rule = "/^-[0-9]*[1-9][0-9]*$/";
        return preg_match($rule, $num);
    }

    /**
     * 非正整数（负整数 + 0）
     *
     * @param string $num
     * @return boolean
     */
    public static function _intz($num)
    {
        $rule = "/^((-\d+)|(0+))$/";
        return preg_match($rule, $num);
    }


    /**
     * 浮点数
     *
     * @param string $num
     * @return boolean
     */
    public static function floats($num)
    {
        $rule = "/^(-?\d+)(\.\d+)?$/";
        return preg_match($rule, $num);
    }


    /**
     * 非负浮点数 ( 正浮点数 )
     *
     * @param string $num
     * @return boolean
     */
    public static function float($num)
    {
        $rule = "/^(([0-9]+\.[0-9]*[1-9][0-9]*)|([0-9]*[1-9][0-9]*\.[0-9]+)|([0-9]*[1-9][0-9]*))$/";
        return preg_match($rule, $num);
    }


    /**
     * 非负浮点数 ( 正浮点数 + 0 )
     *
     * @param string $num
     * @return boolean
     */
    public static function floatz($num)
    {
        $rule = "/^\d+(\.\d+)?$/";
        return preg_match($rule, $num);
    }


    /**
     * 非正浮点数 ( 负浮点数 )
     *
     * @param string $num
     * @return boolean
     */
    public static function _float($num)
    {
        $rule = "/^(-(([0-9]+\.[0-9]*[1-9][0-9]*)|([0-9]*[1-9][0-9]*\.[0-9]+)|([0-9]*[1-9][0-9]*)))$/";
        return preg_match($rule, $num);
    }


    /**
     * 非正浮点数 ( 负浮点数 + 0 )
     *
     * @param string $num
     * @return boolean
     */
    public static function _floatz($num)
    {
        $rule = "/^((-\d+(\.\d+)?)|(0+(\.0+)?))$/";
        return preg_match($rule, $num);
    }


    /******************** 匹配字母 ******************************* */


    /**
     * 由26个英文字母组成的字符串
     *
     * @param string $str
     * @return boolean
     */
    public static function letterAll($str)
    {
        $rule = "/^[A-Za-z]+$/";
        return preg_match($rule, $str);
    }


    /**
     * 由26个英文字母的大写组成的字符串
     *
     * @param string $str
     * @return boolean
     */
    public static function letterB($str)
    {
        $rule = "/^[A-Z]+$/";
        return preg_match($rule, $str);
    }


    /**
     * 由26个英文字母的小写组成的字符串
     *
     * @param string $str
     * @return boolean
     */
    public static function letterS($str)
    {
        $rule = "/^[a-z]+$/";
        return preg_match($rule, $str);
    }



    /********************* 其他 *************************** */


    /**
     * 支付密码
     *
     * @param string $value
     * @return boolean
     */
    public static function payPwd($value)
    {
        $rule = "/^\d{6}$/";
        return preg_match($rule, $value);
    }


    /**
     * 匹配由数字和26个英文字母组成的字符串
     *
     * @param string $value
     * @return boolean
     */
    public static function pwd1($value)
    {
        $rule = "/^[A-Za-z0-9]{6,16}$/";
        return preg_match($rule, $value);
    }


    /**
     * 匹配由数字、26个英文字母或者字符组成的字符串
     *
     * @param string $value
     * @return boolean
     */
    public static function pwd2($value, $minLen = 5, $maxLen = 16)
    {
        $rule = '/^[\\~!@#$%^&*()-_=+|{}\[\],.?\/:;\'\"\d\w]{' . $minLen . ',' . $maxLen . '}$/';
        return preg_match($rule, $value);
    }


    /**
     * 验证用户名
     *
     * @param string $value     验证字符
     * @param integer $minLen   最小长度
     * @param integer $maxLen   最大长度
     * @param string $charset   类型
     * @return boolean
     */
    public static function isNames($value, $minLen = 2, $maxLen = 20, $charset = 'ALL')
    {
        if (empty($value))
            return false;
        switch ($charset) {
            case 'EN':
                // 用户名为大写字母小写字母或下划线,并以字母开头
                $match = '/^[a-za-z][wd_]{' . $minLen . ',' . $maxLen . '}$/iu';
                break;
            case 'CN':
                // 纯中文
                $match = '/^[\x{4e00}-\x{9fa5}]{' . $minLen . ',' . $maxLen . '}$/iu';
                break;
            case 'ALL':
                // 用户名为大写字母小写字母数字或下划线
                $match = '/^[_\x{4e00}-\x{9fa5}a-zA-Z0-9]{' . $minLen . ',' . $maxLen . '}$/iu';
                break;
            default:
                // 任意文本
                $match = '/[\s\S]{' . $minLen . ',' . $maxLen . '}$/iu';
        }
        return preg_match($match, $value);
    }


    /**
     * 验证eamil
     * @param string $value
     * @param int $length
     * @return boolean
     */
    public static function isEmail($value, $match = '/^[\w\d]+[\w\d-.]*@[\w\d-.]+\.[\w\d]{2,10}$/i')
    {
        $v = trim($value);
        if (empty($v))
            return false;
        return preg_match($match, $v);
    }


    /**
     * 手机号
     *
     * @param string $value
     * @return boolean
     */
    public static function isMobile($value)
    {
        $v = trim($value);
        if (empty($v))
            return false;
        $rule = "/^1[3456789]{1}\d{9}$/";
        return preg_match($rule, $value);
    }


    /**
     * 验证邮政编码
     * @param string $value
     * @param string $match
     * @return boolean
     */
    public static function isPostcode($value, $match = '/\d{6}/')
    {
        $v = trim($value);
        if (empty($v))
            return false;
        return preg_match($match, $v);
    }


    /**
     * *
     * 验证URL 
     * @param string $value
     * @param string $match
     * @return boolean
     */
    public static function isURL($value, $match = '/^(http:\/\/)?(https:\/\/)?([\w\d-]+\.)+[\w-]+(\/[\d\w-.\/?%&=]*)?$/')
    {
        $v = strtolower(trim($value));
        if (empty($v))
            return false;
        return preg_match($match, $v);
    }



    //检证身份证是否正确
    public static function isCard($card)
    {
        $card = self::to18Card($card);
        if (strlen($card) != 18) {
            return false;
        }

        $cardBase = substr($card, 0, 17);

        return (self::getVerifyNum($cardBase) == strtoupper(substr($card, 17, 1)));
    }


    //格式化15位身份证号码为18位
    public static function to18Card($card)
    {
        $card = trim($card);

        if (strlen($card) == 18) {
            return $card;
        }

        if (strlen($card) != 15) {
            return false;
        }

        // 如果身份证顺序码是996 997 998 999，这些是为百岁以上老人的特殊编码
        if (array_search(substr($card, 12, 3), array('996', '997', '998', '999')) !== false) {
            $card = substr($card, 0, 6) . '18' . substr($card, 6, 9);
        } else {
            $card = substr($card, 0, 6) . '19' . substr($card, 6, 9);
        }
        $card = $card . self::getVerifyNum($card);
        return $card;
    }


    /**
     * 计算身份证校验码，根据国家标准gb 11643-1999
     *
     * @param string $cardBase 身份证号
     * @return boolean
     */
    private static function getVerifyNum($cardBase)
    {
        if (strlen($cardBase) != 17) {
            return false;
        }
        // 加权因子
        $factor = array(7, 9, 10, 5, 8, 4, 2, 1, 6, 3, 7, 9, 10, 5, 8, 4, 2);

        // 校验码对应值
        $verify_number_list = array('1', '0', 'X', '9', '8', '7', '6', '5', '4', '3', '2');

        $checksum = 0;
        for ($i = 0; $i < strlen($cardBase); $i++) {
            $checksum += substr($cardBase, $i, 1) * $factor[$i];
        }

        $mod = $checksum % 11;
        $verify_number = $verify_number_list[$mod];

        return $verify_number;
    }


    /**
     * 校验金额格式
     * @param  string $accountPrice 金额值
     * @return boolean
     */
    public static function check_money_format($accountPrice)
    {
        if (!preg_match('/^[0-9]+(.[0-9]{1,2})?$/', $accountPrice)) return false;
        return true;
    }
}
