<?php

namespace App\Services\Box;

use App\Models\Box\Box;
use App\Common\Method\Response;
use App\Services\User\UsersModel;
use App\Services\User\FriendModel;
use App\Services\User\CoinLogModel;
use App\Services\User\MoneyLogModel;
use App\Services\User\GoldLogModel;
use App\Models\Admin\Configuration;
use App\Services\Order\OrderModel;
use App\ToolsClass\RedisTool;

/** 拆盒机盒子 */
class BoxModel extends Box
{
    /** 公共条件 */
    public function factor()
    {
        return $this->where(['status' => 1]);
    }

    /**
     * 获取盒子
     *
     * @param  integer $id
     */
    public function getFind($id)
    {
        $box = $this->factor()
            ->where('id', $id)
            ->first();
        if (!$box) {
            return Response::error('该盒子不存在或已下架!');
        }
        return Response::success($box);
    }

    /**
     * 抽奖取消支付
     *
     * @param  object $order
     */
    public static function cancelPay($order)
    {
        return Response::success();
    }

    /**
     * 抽奖支付回调
     *
     * @param  object $order
     * @param  object $user
     */
    public static function notify($order, $user)
    {
        $boxModel = new self();
        $usersModel = new UsersModel();
        $coinLogModel = new CoinLogModel();
        $moneyLogModel = new MoneyLogModel();
        $goldLogModel = new GoldLogModel();
        $friendModel = new FriendModel();
        $box = $boxModel->where('id', $order->data['id'])->first();
        if ($box) {
            // 盒子销量增加
            $box->sales += $order->data['num'];
            $box->save();
            // 抽奖
            $awardModel = new AwardModel();
            $lotteryRes = $awardModel->lottery($order, $box);
            if (!$lotteryRes) {
                return Response::error(Response::getMessage());
            }
            // 邀请赠送积分现金
            // 从 users_tree 表查询邀请关系
            $userTreeModel = new \App\Services\User\TreeModel();
            $userTree = $userTreeModel->where('user_id', $user->id)->first();
            
            // 如果没有邀请关系，直接返回
            if (!$userTree || !$userTree->parent_id) {
                return Response::success();
            }
            
            if ($userTree && $userTree->parent_id) {
                $parent = $usersModel->with('userinfo')->where('id', $userTree->parent_id)->first();
                
                // 如果找不到代理用户，直接返回
                if (!$parent) {
                    \Log::error('找不到代理用户', ['parent_id' => $userTree->parent_id]);
                    return Response::success();
                }
                
                // 确保 userinfo 关系存在
                if (!$parent->userinfo) {
                    \Log::error('代理用户没有 userinfo', ['parent_id' => $parent->id]);
                    return Response::success();
                }
                
                // 检查佣金发放次数限制配置
                $commissionLimitTimes = intval(Configuration::getVal('commission_limit_times') ?? 0);
                // 获取一级分佣比例（百分比）
                $commissionRate = floatval(Configuration::getVal('rebate_one') ?? 0);
                $canSendCommission = true; // 默认可以发放
                
                // 如果配置大于0，需要检查用户已完成的订单次数
                if ($commissionLimitTimes > 0) {
                    // 【第一重保障】先检查该订单是否已经发放过佣金（防止重复发放）
                    $alreadySent = $friendModel
                        ->where('order_id', $order->id)
                        ->exists();
                    
                    if ($alreadySent) {
                        // 该订单已发放过佣金，直接跳过
                        \Log::info('佣金发放跳过：该订单已发放过', [
                            '订单ID' => $order->id,
                            '被邀请人ID' => $user->id,
                        ]);
                        $canSendCommission = false;
                    } else {
                        // 【第二重保障】统计被邀请人获得佣金的次数（从 friend 表统计，基于订单ID）
                        $commissionCount = $friendModel
                            ->where('friend_id', $user->id)   // 被邀请人的ID
                            ->where('order_id', '>', 0)       // 确保有关联订单
                            ->count();
                        
                        // 当前是第几次（从1开始计数）
                        $currentTimes = $commissionCount + 1;
                        
                        \Log::info('佣金限制检查', [
                            '当前订单ID' => $order->id,
                            '被邀请人ID' => $user->id,
                            '被邀请人昵称' => $user->nickName,
                            '代理ID' => $parent->id,
                            '代理昵称' => $parent->nickName,
                            '已发放佣金次数' => $commissionCount,
                            '当前是第几次' => $currentTimes,
                            '佣金发放限制次数' => $commissionLimitTimes,
                            '是否发放佣金' => ($currentTimes <= $commissionLimitTimes) ? '是' : '否'
                        ]);
                        
                        // 如果超过限制次数，不再发放
                        if ($currentTimes > $commissionLimitTimes) {
                            $canSendCommission = false;
                        }
                    }
                }
                
                // 只有满足条件时才发放佣金
                if ($canSendCommission && $parent) {
                    // 计算佣金金额：订单金额 × 分佣比例
                    $orderAmount = $order->pay_money;
                    $commissionAmount = 0;
                    if ($commissionRate > 0 && $orderAmount > 0) {
                        $commissionAmount = calc($orderAmount, '*', $commissionRate / 100, 2);
                    }

                    // 【防并发】同一订单只允许一个请求执行发放，避免微信重复回调导致发两次
                    $lockKey = 'commission_order_' . $order->id;
                    $redis = new RedisTool();
                    
                    $locked = $redis->block($lockKey, 15);
                    if (!$locked) {
                        return Response::success();
                    }
                    
                    try {
                        // 先插入 friend_open_log 占位；若已存在则说明重复回调，直接跳过
                        $friendRes = $friendModel->add($parent, $user, $box, $order, $commissionAmount);
                        if ($friendRes === $friendModel::ADD_RESULT_ALREADY_EXISTS) {
                            return Response::success();
                        }
                        if ($friendRes !== true) {
                            return Response::error('记录添加失败!');
                        }

                        // 再发放 gold / 积分 / 现金
                        if ($commissionAmount > 0) {
                            $remarkText = '邀请用户' . $user->nickName . '开盒赠送';
                            $goldRes = $goldLogModel->log($parent->userinfo, $commissionAmount, $goldLogModel::TYPE_5, $remarkText);
                            if (!$goldRes) {
                                return Response::error('赠送佣金失败!');
                            }
                        }
                        if ($box->send_coin > 0) {
                            $coinRes = $coinLogModel->log($parent->userinfo, calc($box->send_coin, '*', $order->data['num'], 0), $coinLogModel::TYPE_6, '邀请用户' . $user->nickName . '开盒赠送');
                            if (!$coinRes) {
                                return Response::error('赠送积分失败!');
                            }
                        }
                        if ($box->send_money > 0) {
                            $moneyRes = $moneyLogModel->log($parent->userinfo, calc($box->send_money, '*', $order->data['num'], 2), $moneyLogModel::TYPE_9, '邀请用户' . $user->nickName . '开盒赠送');
                            if (!$moneyRes) {
                                return Response::error('赠送现金失败!');
                            }
                        }
                    } finally {
                        $redis->unlock($lockKey);
                    }
                }
            }
        }
        return Response::success();
    }
}
