<?php

namespace App\Http\Controllers\Api\V1\Shop;

use Exception;
use App\ToolsClass\RedisTool;
use App\ToolsClass\RegexTool;
use App\Common\Enum\RedisEnum;
use App\Common\Method\Response;
use App\Services\Shop\SkuModel;
use App\Services\Shop\GoodsModel;
use App\Services\Order\OrderModel;
use Illuminate\Support\Facades\DB;
use App\Services\User\AddressModel;
use App\Services\Order\OrderListModel;
use App\Http\Controllers\BaseController;

// 订单
class OrderController extends BaseController
{
    /**
     * 确认、提交订单
     *
     * @param  \App\Services\Order\OrderModel     $orderModel
     * @param  \App\ToolsClass\RedisTool          $redis
     * @param  \App\Services\Shop\GoodsModel      $goodsModel
     * @param  \App\Services\Shop\SkuModel        $skuModel
     * @param  \App\ToolsClass\RegexTool          $regexTool
     * @param  \App\Services\User\AddressModel    $addressModel
     * @param  \App\Services\Order\OrderListModel $orderListModel
     */
    public function order(
        OrderModel $orderModel,
        RedisTool $redis,
        GoodsModel $goodsModel,
        SkuModel $skuModel,
        RegexTool $regexTool,
        AddressModel $addressModel,
        OrderListModel $orderListModel
    ) {
        $user = auth('api')->user();
        $submit = $this->request->submit;
        // 商品数据
        $goods = $goodsModel->factor()->where('id', $this->request->id)->first();
        if (!$goods) {
            return $this->error('商品不存在或已下架!');
        }
        // 地址判断 运费计算
        $express_money = 0;
        if ($submit == 1 || $this->request->add_id) {
            $add = $addressModel->where(['id' => $this->request->add_id, 'user_id' => $user->id])->first();
            if (!$add) {
                return $this->error('地址选择错误!');
            }
            $express_money = $orderListModel->fareCale($add);   // 运费计算
        }
        //  支付方式
        if (!in_array($this->request->pay_type, [$orderModel::P_TYPE_3])) {
            return $this->error('支付方式选择有误!');
        }
        // 数量验证
        $num = $this->request->num;
        if (!$regexTool->int($num)) {
            return $this->error('数量填写不正确!');
        }
        $key = RedisEnum::SHOP . $goods->id;
        if ($redis->block($key)) {
            // 规格数据
            $sku = $skuModel->where(['id' => $this->request->sku_id, 'goods_id' => $goods->id, 'status' => 1])->first();
            if (!$sku) {
                return $this->unlock($redis, $key)->error('规格不存在,或已下架!');
            }
            // 判断库存
            if ($num > $sku->stock) {
                return $this->unlock($redis, $key)->error('库存不足!');
            }
            $data['goods']['id'] = $goods->id;
            $data['goods']['title'] = $goods->title;
            $data['goods']['thumb'] = $goods->thumb[0];
            $data['goods']['price'] = $sku->price;
            $data['goods']['num'] = $num;
            $data['goods']['sku_name'] = $sku->name;
            $total_money = calc($sku->price * $num, '+', $express_money);   // 商品价格+运费
            // 抵扣计算
            $param = [
                'pay_type' => $this->request->pay_type,
                'gold' => true,
            ];
            $discount_data = $orderModel->discount($user, $total_money, $param);
            if (!$discount_data) {
                return $this->unlock($redis, $key)->error(Response::getMessage());
            }
            // 确认订单
            if ($submit != 1) {
                $data['discount'] = $discount_data;
                $data['express']['express_money'] = $express_money;
                return $this->unlock($redis, $key)->success('请求成功', $data);
            }
            // 开启事务提交
            DB::beginTransaction();
            try {
                // 总销量库存&规格销量增加库存减少
                $goods->sales += $num;
                $goods->stock -= $num;
                $sku->sales += $num;
                $sku->stock -= $num;
                if (!$goods->save() || !$sku->save()) {
                    DB::rollBack();
                    return $this->unlock($redis, $key)->error('下单失败!');
                }
                // 组合支付数据
                $data = [
                    'id' => $goods->id,
                    'title' => $goods->title,
                    'thumb' => imageUrl($goods->thumb[0], false),
                    'price' => $sku->price,
                    'num' => $num,
                    'sku_id' => $sku->id,
                    'sku_name' => $sku->name,
                    'express_money' => $express_money,
                    'name' => $add->name,
                    'mobile' => $add->mobile,
                    'address' => $add->address,
                    'ramk' => $this->request->ramk ?? '',
                ];
                $pay = $orderModel->pay($user, $orderModel::TYPE_3, $param, $discount_data, $data);
                if (!$pay) {
                    DB::rollBack();
                    return $this->unlock($redis, $key)->error(Response::getMessage());
                }
                $returnData = [
                    'order_sn' => $pay->order_sn,
                    'pay_money' => $pay->pay_money,
                    'pay_type' => $pay->pay_type,
                ];
                DB::commit();
                return $this->unlock($redis, $key)->success('下单成功!', $returnData);
            } catch (Exception $e) {
                DB::rollBack();
                return $this->unlock($redis, $key)->error('操作失败!');
            }
        }
        return $this->error('下单失败,请重试!');
    }
}
