<?php

namespace App\Http\Controllers\Api\V1\Ship;

use Exception;
use App\ToolsClass\RedisTool;
use App\Common\Enum\RedisEnum;
use App\Common\Method\Response;
use App\ToolsClass\AliwuliuTool;
use App\Services\Box\RatingModel;
use App\Services\Box\AwardModel;
use App\Services\Order\OrderModel;
use App\Services\Shop\ChangeModel;
use Illuminate\Support\Facades\DB;
use App\Services\User\AddressModel;
use App\Services\Order\OrderListModel;
use App\Http\Controllers\BaseController;
use App\Http\Requests\Api\Ship\RatingRequest;
use App\Http\Resources\Ship\ShipInfoResources;
use App\Http\Resources\User\AwardListResources;
use App\Http\Resources\Ship\ExpressInfoResources;
use App\Services\User\MoneyLogModel;
use App\Models\Admin\Configuration;
class BoxController extends BaseController
{
    /**
     * 发货确认，提交
     *
     * @param  \App\Services\Order\OrderListModel $orderListModel
     * @param  \App\Services\Order\OrderModel     $orderModel
     * @param  \App\Services\User\AddressModel    $addressModel
     * @param  \App\ToolsClass\RedisTool          $redis
     */
    public function order(
        OrderListModel $orderListModel,
        OrderModel $orderModel,
        AddressModel $addressModel,
        RedisTool $redis,
        ChangeModel $changeModel
    ) {
        $user = auth('api')->user();
        $submit = $this->request->submit;
        // Redis 锁
        $key = RedisEnum::SHIP . $user->id;
        if ($redis->block($key)) {
            //  支付方式
            if (!in_array($this->request->pay_type, [$orderModel::P_TYPE_3])) {
                return $this->unlock($redis, $key)->error('支付方式选择有误!');
            }
            // 地址判断 运费计算
            $express_money = 0;
            if ($submit == 1 || $this->request->add_id) {
                $add = $addressModel->where(['id' => $this->request->add_id, 'user_id' => $user->id])->first();
                if ($submit == 1 && !$add) {
                    return $this->unlock($redis, $key)->error('地址选择错误!');
                }
                $express_money = $orderListModel->fareCale($add);
            }
            // 判断是直接发货还是更换商品时候发货
            $type = $this->request->type;
            if ($type == 1) {
                // 盒柜直接发货
                $where = [
                    ['id', '=', $this->request->id],
                    ['user_id', '=', $user->id],
                    ['status', '=', $orderListModel::STA_0],
                ];
                $info = $orderListModel->where($where)
                    ->whereIn('type', [$orderListModel::TYPE_1, $orderListModel::TYPE_2])
                    ->first();
                if (!$info) {
                    return $this->unlock($redis, $key)->error('选择的奖品不存在或已发货,请重新选择!');
                }
                $data['data']['id'] = $info->id;
                $data['data']['title'] = $info->title;
                $data['data']['thumb'] = $info->thumb;
                $data['data']['price'] = $info->price;
                $data['data']['num'] = $info->num;
            } elseif ($type == 2) {
                // 置换商品判断
                $change = $changeModel->factor()->where('id', $this->request->change_id)->first();
                if (!$change) {
                    return $this->unlock($redis, $key)->error('请选择置换商品,或置换商品选择不正确!');
                }
                // 更换商品发货
                $where = [
                    ['id', '=', $this->request->id],
                    ['user_id', '=', $user->id],
                    ['status', '=', $orderListModel::STA_0],
                    ['type', '=', $orderListModel::TYPE_1],
                    ['is_change', '=', 0],
                ];
                $info = $orderListModel->where($where)->first();
                if (!$info) {
                    return $this->unlock($redis, $key)->error('选择的奖品不存在或已发货,请重新选择!');
                }
                $data['old']['id'] = $info->id;
                $data['old']['title'] = $info->title;
                $data['old']['thumb'] = $info->thumb;
                $data['old']['price'] = $info->price;
                $data['old']['num'] = $info->num;
                $data['new']['id'] = $change->id;
                $data['new']['title'] = $change->title;
                $data['new']['thumb'] = $change->thumb[0];
                $data['new']['price'] = $change->price;
                $data['new']['num'] = 1;
            } else {
                return $this->unlock($redis, $key)->error('参数有误!');
            }
            // 抵扣计算
            $param = [
                'pay_type' => $this->request->pay_type,
                'money' => true,
            ];
            $discount_data = $orderModel->discount($user, $express_money, $param);
            if (!$discount_data) {
                return $this->unlock($redis, $key)->error(Response::getMessage());
            }
            // 确认订单
            if ($submit != 1) {
                $data['discount'] = $discount_data;
                $data['express']['express_money'] = $express_money;
                return $this->unlock($redis, $key)->success('请求成功', $data);
            }
            // 开启事务
            DB::beginTransaction();
            try {
                if ($type == 1) {
                    // 组合支付数据
                    $data = [
                        'type' => 1,
                        'order_list_id' => $info->id,
                        'express_money' => $express_money,
                        'name' => $add->name,
                        'mobile' => $add->mobile,
                        'address' => $add->address,
                        'ramk' => $this->request->ramk ?? '',
                    ];
                }
                if ($type == 2) {
                    // 组合支付数据
                    $data = [
                        'type' => 2,
                        'order_list_id' => $info->id,
                        'express_money' => $express_money,
                        'name' => $add->name,
                        'mobile' => $add->mobile,
                        'address' => $add->address,
                        'ramk' => $this->request->ramk ?? '',
                        'change_id' => $change->id,
                        'change_title' => $change->title,
                        'change_thumb' => $change->thumb[0],
                        'change_price' => $change->price,
                    ];
                }
                // 修改状态
                $info->status = $orderListModel::STA_1;
                if (!$info->save()) {
                    DB::rollBack();
                    return $this->unlock($redis, $key)->error('操作失败!');
                }
                $pay = $orderModel->pay($user, $orderModel::TYPE_2, $param, $discount_data, $data);
                if (!$pay) {
                    DB::rollBack();
                    return $this->unlock($redis, $key)->error(Response::getMessage());
                }
                $returnData = [
                    'order_sn' => $pay->order_sn,
                    'pay_money' => $pay->pay_money,
                    'pay_type' => $pay->pay_type,
                ];
                DB::commit();
                return $this->unlock($redis, $key)->success('提交成功!', $returnData);
            } catch (Exception $e) {
                DB::rollBack();
                return $this->unlock($redis, $key)->error('提交失败!');
            }
        }
        return $this->error('操作过快!');
    }

    /**
     * 盒柜
     *
     * @param  \App\Services\Order\OrderListModel $model
     */
    public function list(OrderListModel $model)
    {
        $user = auth('api')->user();
        // 筛选
        $where = [
            ['user_id', '=', $user->id],
            ['status', '=', $this->request->status ?? 0],
        ];
        if ($this->request->mark_id) {
            $where[] = ['mark_id', '=', $this->request->mark_id ?? 0];
        }
        $list = $model->where($where)
            ->whereIn('type', [$model::TYPE_1, $model::TYPE_2])
            ->orderByRaw('id desc')
            ->paginate($this->request->per_page ?? 20);
        $list = $model->page($list);
        $list['data'] = AwardListResources::collection($list['data']);
        return $this->success('请求成功', $list);
    }

    /**
     * 发货详情
     *
     * @param  \App\Services\Order\OrderListModel $model
     */
    public function info(OrderListModel $model)
    {
        $user = auth('api')->user();
        $where = [
            ['id', '=', $this->request->id],
            ['user_id', '=', $user->id],
        ];
        $data = $model->where($where)
            ->whereIn('type', [$model::TYPE_1, $model::TYPE_2])
            ->first();
        if (!$data) {
            return $this->error('参数错误!');
        }
        $data = new ShipInfoResources($data);
        return $this->success('请求成功', $data);
    }

    /**
     * 发货详情
     *
     * @param  \App\Services\Order\OrderListModel $model
     */
    public function express(OrderListModel $model, AliwuliuTool $aliwuliuTool)
    {
        $user = auth('api')->user();
        $where = [
            ['id', '=', $this->request->id],
            ['user_id', '=', $user->id],
            ['status', '>=', $model::STA_2],
        ];
        $data = $model->where($where)
            ->whereIn('type', [$model::TYPE_1, $model::TYPE_2])
            ->first();
        if (!$data) {
            return $this->error('参数错误!');
        }
        // 快递信息
        $wuliu = $aliwuliuTool->search($data->shipinfo->express_num, $data->shipinfo->express_code, $data->shipinfo->mobile);
        if (!$wuliu) {
            return $this->error(Response::getMessage());
        }
        $data['express'] = $wuliu;
        $data = new ExpressInfoResources($data);
        return $this->success('请求成功', $data);
    }

    /**
     * 退货商品
     *
     * @param  \App\Services\Order\OrderListModel $model
     */
    public function dissolve(OrderListModel $model,MoneyLogModel $moneyLogModel)
    {
        $configModel = new Configuration();
        $awardModel = new AwardModel();
        $user = auth('api')->user();

        $where = [
            ['user_id', '=', $user->id],
            ['status', '=', $model::STA_0],
        ];
        if (is_array($this->request->id)) {
            $data = $model->where($where)
            ->whereIn('id',$this->request->id)
            ->whereIn('type', [$model::TYPE_1, $model::TYPE_2])
            ->get();
            if (!$data) {
                return $this->error('参数错误!');
            }
            $money = 0;
            foreach ($data as &$value){
                $rate = $awardModel->where('id',$value->goods_id)->first();
                $money += bcmul($value->price,bcdiv($rate->dissolve_rate,100,2),2);
                
            }
            $res = $model->whereIn('id',$this->request->id)->whereIn('type', [$model::TYPE_1, $model::TYPE_2])->update(['status'=>$model::STA_5]);
            $moneyLogModel->log($user->userinfo,$money,$moneyLogModel::TYPE_11);
            if (empty($res)) {
                return $this->error('操作失败!');
            }
            return $this->success('退货成功!');
        }else{
            $where[] = ['id','=',$this->request->id];
            $data = $model->where($where)
            ->whereIn('type', [$model::TYPE_1, $model::TYPE_2])
            ->first();
            if (!$data) {
                return $this->error('参数错误!');
            }
            
            // 更改订单状态
            if ($data->type == $model::TYPE_1) {
                $data->status = $model::STA_5;
            }
            $rate = $awardModel->where('id',$data->goods_id)->first();
            $money = bcmul($data->price,bcdiv($rate->dissolve_rate,100,2),2);
            $moneyLogModel->log($user->userinfo,$money,$moneyLogModel::TYPE_11);
            if (!$data->save()) {
                return $this->error('操作失败!');
            }
            return $this->success('退货成功!');
        }
        
    }
    /**
     * 确认收货
     *
     * @param  \App\Services\Order\OrderListModel $model
     */
    public function receipt(OrderListModel $model)
    {
        $user = auth('api')->user();
        $where = [
            ['id', '=', $this->request->id],
            ['user_id', '=', $user->id],
            ['status', '=', $model::STA_2],
        ];
        $data = $model->where($where)
            ->whereIn('type', [$model::TYPE_1, $model::TYPE_2])
            ->first();
        if (!$data) {
            return $this->error('参数错误!');
        }
        // 抽奖奖品
        if ($data->type == $model::TYPE_1) {
            $data->status = $model::STA_3;
        }
        // 抽奖的置换商品/抽奖的稀有普通奖品/天天抽奖奖品（直接完成，略过评价步骤）
        if (($data->type == $model::TYPE_1 && $data->is_change == 1) || ($data->type == $model::TYPE_1 && in_array($data->mark_id, [3, 4])) || ($data->type == $model::TYPE_2)) {
            $data->status = $model::STA_4;
        }
        if (!$data->save()) {
            return $this->error('操作失败!');
        }
        return $this->success('操作成功!');
    }

    /**
     * 评价
     *
     * @param  \App\Services\Order\OrderListModel $model
     */
    public function rating(
        OrderListModel $model,
        RatingRequest $validate,
        RatingModel $ratingModel
    ) {
        // 验证
        if (!$validate->check($this->request->all())) {
            return $this->error($validate->getError());
        }
        $user = auth('api')->user();
        $where = [
            ['id', '=', $this->request->id],
            ['user_id', '=', $user->id],
            ['status', '=', $model::STA_3],
            ['type', '=', $model::TYPE_1],
        ];
        $data = $model->where($where)->first();
        if (!$data) {
            return $this->error('参数错误!');
        }
        DB::beginTransaction();
        try {
            // 状态修改
            $data->status = $model::STA_4;
            // 评价
            $log = $ratingModel;
            $log->box_id = $data->data_id;
            $log->user_id = $user->id;
            $log->goods_id = $data->goods_id;
            $log->title = $data->title;
            $log->star = $this->request->star;
            $log->content = $this->request->content;
            $log->thumb = $this->request->thumb;
            $log->is_hidden = $this->request->is_hidden ?? 2;
            if (!$data->save() || !$log->save()) {
                DB::rollBack();
                return $this->error('评价失败!');
            }
            DB::commit();
            return $this->success('评价成功!');
        } catch (Exception $e) {
            DB::rollBack();
            return $this->error('操作失败!');
        }
    }
}
