<?php

namespace App\Http\Controllers\Api\V1\Box;

use Exception;
use App\ToolsClass\RedisTool;
use App\Common\Enum\RedisEnum;
use App\Services\Box\BoxModel;
use App\Common\Method\Response;
use App\Services\Box\DrawModel;
use App\Services\Box\AwardModel;
use App\Services\Order\OrderModel;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\BaseController;
use App\Http\Resources\Box\TryPlayResources;

// 订单
class OrderController extends BaseController
{
    /**
     * //查询订单状态
     * @param OrderModel $orderModel
     * @return bool|JsonResponse
     */
    public function orderStatus(OrderModel $orderModel)
    {
        $order_sn = $this->request->order_sn;
        $res =  $orderModel->getOrderStatus($order_sn);
        if(!$res){
            return $this->success('请求成功', ['res'=>'no']);
        }
        return $this->success('请求成功', ['res'=>'ok']);
    }

    /**
     * 确认、提交订单
     *
     * @param OrderModel $model
     * @param BoxModel $boxModel
     * @param DrawModel $drawModel
     * @param RedisTool $redis
     * @return JsonResponse
     * @throws Exception
     */
    public function order(
        OrderModel $model,
        BoxModel $boxModel,
        DrawModel $drawModel,
        RedisTool $redis
    ) {
        $user = auth('api')->user();
        $key = RedisEnum::BOX_DRAW . $user->id;
        if ($redis->block($key)) {
            //  支付方式
            if (!in_array($this->request->pay_type, [$model::P_TYPE_3,$model::P_TYPE_4])) {
                return $this->unlock($redis, $key)->error('支付方式选择有误!');
            }
            // 盒子数据
            $box = $boxModel->getFind($this->request->id);
            if (!$box) {
                return $this->unlock($redis, $key)->error(Response::getMessage());
            }
            // 抽取方式
            $draw = $drawModel->factor()->where('id', $this->request->draw_id)->first();
            if (!$draw) {
                return $this->unlock($redis, $key)->error('抽取方式错误!');
            }
            $num = $draw->times;
            $data['box']['id'] = $box->id;
            $data['box']['title'] = $box->title;
            $data['box']['price'] = $box->price;
            $data['box']['num'] = $num;
            $total_money = calc($box->price, '*', $num);
            // 抵扣计算
            $param = [
                'pay_type' => $this->request->pay_type,
                'money' => true,
            ];
            $discount_data = $model->discount($user, $total_money, $param);
            if (!$discount_data) {
                return $this->unlock($redis, $key)->error(Response::getMessage());
            }
            // 确认订单
            if ($this->request->submit != 1) {
                $data['discount'] = $discount_data;
                return $this->unlock($redis, $key)->success('请求成功', $data);
            }
            // 开启事务提交
            DB::beginTransaction();
            try {
                // 组合支付数据
                $data = [
                    'id' => $box->id,
                    'title' => $box->title,
                    'thumb' => imageUrl($box->thumb, false),
                    'price' => $box->price,
                    'num' => $num,
                    'invite_user_id' => $this->request->invite_user_id,
                    'draw_id' => $draw->id,
                ];
                $pay = $model->pay($user, $model::TYPE_1, $param, $discount_data, $data);
                if (!$pay) {
                    DB::rollBack();
                    return $this->unlock($redis, $key)->error(Response::getMessage());
                }
                $returnData = [
                    'order_sn' => $pay->order_sn,
                    'pay_money' => $pay->pay_money,
                    'pay_type' => $pay->pay_type,
                ];
                DB::commit();
                
                return $this->unlock($redis, $key)->success('下单成功!', $returnData);
            } catch (Exception $e) {
                DB::rollBack();
                return $this->unlock($redis, $key)->error('下单失败!');
            }
        }
        return $this->error('下单失败,请重试!');
    }

    /**
     * 试玩盒子
     *
     * @param  \App\Services\Box\BoxModel   $boxModel
     * @param  \App\Services\Box\AwardModel $awardModel
     */
    public function tryPlay(
        BoxModel $boxModel,
        AwardModel $awardModel,
        DrawModel $drawModel
    ) {
        // 盒子数据
        $box = $boxModel->getFind($this->request->id);
        if (!$box) {
            return $this->error(Response::getMessage());
        }
        // 抽取方式
        $draw = $drawModel->factor()->inRandomOrder()->take(1)->first();
        if (!$draw) {
            return $this->error('抽取方式错误!');
        }
        $list = $awardModel->where(['box_id' => $box->id])->inRandomOrder()->take($draw->times)->get();
        $list = TryPlayResources::collection($list);
        return $this->success('请求成功', $list);
    }
}
