<?php

namespace App\Http\Controllers\Admin\Admin;

use App\Models\Admin\Role;
use App\Models\Admin\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\URL;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\View;
use App\Http\Controllers\BaseController;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Support\Facades\Response;
use Illuminate\Support\Facades\Validator;
use Illuminate\Foundation\Auth\AuthenticatesUsers;
use App\Http\Requests\Admin\Admin\UserCreateRequest;
use App\Http\Requests\Admin\Admin\ChangePasswordRequest;

class UserController extends BaseController
{
    use AuthenticatesUsers;

    /** 实例 */
    protected $model;

    /** 初始化 */
    public function __construct()
    {
        parent::__construct();
        // 初始化实例
        $this->model = new User();
    }

    /**
     * Get the guard to be used during authentication.
     *
     * @return \Illuminate\Contracts\Auth\StatefulGuard
     */
    protected function guard()
    {
        return Auth::guard('admin');
    }

    /**
     * 用户登录表单
     * @return \Illuminate\Contracts\View\View
     */
    public function showLoginForm()
    {
        return View::make('admin.admin.user.login');
    }

    //登录成功后的跳转地址
    public function redirectTo()
    {
        return URL::route('admin.layout');
    }

    /** 
     * 登录 
     */
    public function login()
    {
        // $admin = $this->model->find(1);
        // $admin->password = 123456;
        // $admin->save();die;
        // 规则
        $rules = [
            'username' => 'required',
            'password' => 'required',
            'captcha'  => 'required|captcha'
        ];

        // 自定义消息
        $messages = [
            'username.required' => '请输入用户名',
            'password.required' => '请输入密码',
            'captcha.required'  => '请输入验证码',
            'captcha.captcha'   => '验证码不正确',
        ];

        $validator = Validator::make($this->request->all(), $rules, $messages);
        if ($validator->fails()) {
            return $this->error($validator->errors()->first());
        }
        //登录
        $username = $this->request->username;
        $user = $this->model->where('username', $username)->first();
        if (!$user) {
            return $this->error('该用户不存在');
        }
        $password = $this->request->password;
        if (!auth('admin')->attempt(['username' => $username, 'password' => $password])) {
            return $this->error('密码不正确');
        }
        //登录之后切记不能有任何输出
        return $this->success('登录成功');
    }

    /**
     * 退出后的动作
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    protected function loggedOut(Request $request)
    {
        return Redirect::to(URL::route('admin.login'));
    }

    /**
     * 更改密码
     * @return \Illuminate\Contracts\View\View
     */
    public function changeMyPasswordForm()
    {
        return View::make('admin.admin.user.changeMyPassword');
    }

    /**
     * 修改密码
     *
     * @param  \App\Http\Requests\Admin\Admin\ChangePasswordRequest $request
     */
    public function changeMyPassword(ChangePasswordRequest $validate)
    {
        $user = auth('admin')->user();
        $user = $this->model->find($user->id);
        $data = $this->request->all();
        if (!$validate->check($data)) {
            return $this->error($validate->getError());
        }
        //验证原密码
        if (!Hash::check($data['old_password'], $user->getAuthPassword())) {
            return $this->error('原密码不正确');
        }
        try {
            $user->password = $data['new_password'];
            $user->save();
            return $this->success('密码修改成功!');
        } catch (\Exception $exception) {
            return $this->error('修改失败!');
        }
    }

    /**
     * 用户列表主页
     * @return \Illuminate\Contracts\View\View
     */
    public function index()
    {
        return View::make('admin.admin.user.index');
    }

    /**
     * 用户数据
     *
     */
    public function data()
    {
        // 筛选
        $where = [];
        // 状态筛选
        if ($this->request->username) {
            $where[] = ['username', 'like', '%' . $this->request->username . '%'];
        }
        $res = $this->model->where($where)->paginate($this->request->get('limit', 30));
        $data = [
            'code' => 0,
            'msg' => '正在请求中...',
            'count' => $res->total(),
            'data' => $res->items()
        ];
        return Response::json($data);
    }

    /**
     * 添加用户
     * @return \Illuminate\Contracts\View\View
     */
    public function create()
    {
        $model = $this->model;
        return View::make('admin.admin.user.create', compact('model'));
    }

    /**
     * 添加用户
     * @param UserCreateRequest $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(UserCreateRequest $validate)
    {
        $data = $this->request->all();
        if (!$data['password']) {
            return $this->error('请填写密码');
        }
        if (!$validate->check($data)) {
            return $this->error($validate->getError());
        }
        try {
            $this->model->create($data);
            return $this->success('添加成功');
        } catch (\Exception $exception) {
            return $this->error('添加失败!');
        }
    }

    /**
     * 更新用户
     * @return \Illuminate\Contracts\View\View
     */
    public function edit()
    {
        $model = $this->model->findOrFail($this->request->id);
        return View::make('admin.admin.user.edit', compact('model'));
    }

    /**
     * 更新用户
     * @param UserCreateRequest $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(UserCreateRequest $validate)
    {
        $user = $this->model->findOrFail($this->request->id);
        $data = $this->request->all();
        if (!$validate->check($data)) {
            return $this->error($validate->getError());
        }
        // 不填密码
        if (!$this->request->password) {
            unset($data['password']);
        }
        try {
            $user->update($data);
            return $this->success('更新成功!');
        } catch (\Exception $exception) {
            return $this->error('更新失败!');
        }
    }

    /**
     * 删除用户
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy()
    {
        $ids = $this->request->ids;
        if (!is_array($ids) || empty($ids)) {
            return $this->error('请选择删除项');
        }
        // Admin 用户不可删除
        if (in_array(1, $ids)) {
            return $this->error('包含用户ID1不可删除,请重新选择');
        }
        try {
            $this->model->destroy($ids);
            return $this->success('删除成功!');
        } catch (\Exception $e) {
            return $this->error('删除失败');
        }
    }

    /**
     * 分配角色
     * @return \Illuminate\Contracts\View\View
     */
    public function role()
    {
        $user = $this->model->findOrFail($this->request->id);
        $roles = Role::get();
        foreach ($roles as $role) {
            $role->own = $user->hasRole($role) ? true : false;
        }
        return View::make('admin.admin.user.role', compact('roles', 'user'));
    }

    /**
     * 分配角色
     * @return \Illuminate\Http\RedirectResponse
     */
    public function assignRole()
    {
        $id = $this->request->id;
        // Admin 用户角色不可编辑
        if ($id == 1) {
            return $this->error('该用户不可编辑角色');
        }
        $user = $this->model->findOrFail($id);
        $roles = $this->request->get('roles', []);
        try {
            $user->syncRoles($roles);
            return $this->success('更新成功');
        } catch (\Exception $exception) {
            return $this->error('更新失败!');
        }
    }
}
