<?php

/** ------------------------- String 字符串处理 ------------------------------- */

if (!function_exists('jsonDecode')) {
    /**
     * Json 字符串转数组
     *
     * @param  string $str
     * @return array
     */
    function jsonDecode($str)
    {
        // 防止PHP函数 magic_quotes_gpc=On 开起再次转义问题
        if (ini_get("magic_quotes_gpc") == "1") {
            $str = stripslashes($str);
        }
        return json_decode($str, true);
    }
}

if (!function_exists('jsonEncode')) {
    /**
     * 数组转Json 字符串
     *
     * @param  array $data
     * @return array
     */
    function jsonEncode($data)
    {
        return json_encode($data, JSON_UNESCAPED_UNICODE);
    }
}

if (!function_exists('imageUrl')) {

    /**
     * 处理图图片
     *
     * @param  $data    图片数据
     * @param  $status  处理方式：true 添加域名，false 去掉域名
     */
    function imageUrl($data, $status = true)
    {
        $type = '';
        if (is_string($data)) {
            // 字符串
            $new_data = json_decode($data, true);
            if ($new_data && is_array($new_data) && !empty($new_data)) {
                // Json 字符串
                $type = 2;
            } else {
                // 字符串，拼接
                $new_data = explode(',', $data);
                $type = 1;
            }
        } else {
            // 数组
            $new_data = $data;
        }
        // 循环处理数据
        if ($new_data && is_array($new_data) && !empty($new_data)) {
            $data = [];
            foreach ($new_data as $k => $v) {
                $data[$k] = editImageUrl($v, $status);
            }
        }
        // 处理返回数据
        if ($type == 1) {
            // 字符串，拼接
            $data = implode(',', $data);
        } elseif ($type == 2) {
            // Json 字符串
            $data = json_encode($data, true);
        }
        return $data;
    }
}
if (!function_exists('editImageUrl')) {
    /**
     * 处理图片Http 域名
     *
     * @param string    $url    Url 路径
     * @param bool      $status 处理方式：true 添加，false 去掉 域名
     * @return string
     */
    function editImageUrl($url, $status)
    {
        if ($url) {
            // exit;
            if ($status) {
                // 保存老的地址
                $old_url = $url;
                // 去掉Http 域名
                $data = parse_url($url);
                if (isset($data['path'])) {
                    $image = parse_url($url)['path'];
                    $url = $image;
                }
                // 处理添加域名
                // print_r($url);
                // exit;
                if (strstr($url, '/uploads/')) {
                    // 本地域名拼接
                    $url = env('APP_URL') . $url;
                } elseif (strstr($url, '/aliyun/')) {
                    // echo 111;
                    // exit;
                    //阿里云域名拼接
                    $url = env('OSS_URL') . $url;
                } elseif (strstr($url, '/qiniuyun/')) {
                    //七牛云域名拼接
                    $url = env('QINIU_URL') . $url;
                } else {
                    $url = $old_url;
                }
            } else {
                //去除Http 域名
                if (strstr($url, 'http')) {
                    $data = parse_url($url);
                    if (isset($data['path'])) {
                        $image = parse_url($url)['path'];
                        $url = $image;
                    }
                }
            }
        }
        return $url;
    }
}

if (!function_exists('getSrc')) {
    /**
     * 富文本编辑器路径获取
     *
     * @param  string $contentStr
     */
    function getSrc($contentStr)
    {
        $imgSrcArr = [];
        //首先将富文本字符串中的 img 标签进行匹配
        $pattern_imgTag = '/<img\b.*?(?:\>|\/>)|<video\b.*?(?:\>|\/>)|<audio\b.*?(?:\>|\/>)|<source\b.*?(?:\>|\/>)/i';
        preg_match_all($pattern_imgTag, $contentStr, $matchIMG);
        if (isset($matchIMG[0])) {
            foreach ($matchIMG[0] as $key => $imgTag) {
                //进一步提取 img标签中的 src属性信息
                $pattern_src = '/\bsrc\b\s*=\s*[\'\"]?([^\'\"]*)[\'\"]?/i';
                preg_match_all($pattern_src, $imgTag, $matchSrc);
                if (isset($matchSrc[1])) {
                    foreach ($matchSrc[1] as $src) {
                        //将匹配到的src信息压入数组
                        $imgSrcArr[] = $src;
                    }
                }
            }
        }
        $imgSrcArr = array_unique($imgSrcArr);
        return $imgSrcArr;
    }
}

if (!function_exists('formatPrice')) {

    /**
     * 格式化金额
     *
     * @param string $money     需要格式化的金额
     * @param integer $decimals 格式化保留的位数
     * @return float            返回的金额
     */
    function formatPrice($money, $decimals = 2)
    {
        if (strstr($money, '+')) {
            $format = '+';
        } elseif (strstr($money, '-')) {
            $format = '-';
        } else {
            $format = '';
        }
        $money = bcadd(abs($money), 0, $decimals);
        $money = $format . $money;
        return $money;
    }
}

if (!function_exists('calc')) {

    /**
     * 金额计算（加减乘除...）
     *
     * @param string $n1        需要计算的数1
     * @param string $symbol    计算符号
     * @param string $n2        需要计算的数2
     * @param string $scale     格式化保留的位数
     * @return float            返回的金额
     */
    function calc($n1, $symbol, $n2, $scale = '2')
    {
        $res = "";
        switch ($symbol) {
            case "+": //加法
                $res = bcadd($n1, $n2, $scale);
                break;
            case "-": //减法
                $res = bcsub($n1, $n2, $scale);
                break;
            case "*": //乘法
                $res = bcmul($n1, $n2, $scale);
                break;
            case "/": //除法
                $res = bcdiv($n1, $n2, $scale);
                break;
            case "%": //求余、取模
                $res = bcmod($n1, $n2, $scale);
                break;
            default:
                $res = "";
                break;
        }
        return $res;
    }
}

if (!function_exists('nonceStr')) {

    /**
     * 生成随机字符串
     *
     * @param integer $length
     * @return void
     */
    function nonceStr($length = 16)
    {
        $chars = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789";
        $str = "";
        for ($i = 0; $i < $length; $i++) {
            $str .= substr($chars, mt_rand(0, strlen($chars) - 1), 1);
        }
        return $str;
    }
}

if (!function_exists('orderNum')) {

    /**
     * 生成订单号
     *
     */
    function orderNum()
    {
        $yCode = array('A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z');
        $orderSn = $yCode[intval(date('Y')) - 2021] . strtoupper(dechex(date('m'))) . date('d') . substr(time(), -5) . substr(microtime(), 2, 5) . sprintf('%02d', mt_rand(0, 99));
        return $orderSn;
    }
}

if (!function_exists('code')) {
    /**
     * 邀请码、兑换码生成
     * @param $length 长度不得超过8
     * @return void
     */
    function code($length = 8)
    {
        $code = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ';
        $rand = $code[mt_rand(0, 25)]
            . strtoupper(dechex(date('m')))
            . date('d')
            . substr(time(), -5)
            . substr(microtime(), 2, 5)
            . sprintf('%02d', mt_rand(0, 99));
        for (
            $a = md5($rand, true),
            $s = '0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ',
            $d = '',
            $f = 0;
            $f < $length;
            $g = ord($a[$f]),
            $d .= $s[($g ^ ord($a[$f + 8])) - $g & 0x1F],
            $f++
        );
        return $d;
    }
}

if (!function_exists('randomPhone')) {

    /**
     * 生成随机手机号
     *
     */
    function randomPhone()
    {
        $tel_arr = array(
            '130', '131', '132', '133', '134', '135', '136', '137', '138', '139', '144', '147', '150', '151', '152', '153', '155', '156', '157', '158', '159', '176', '177', '178', '180', '181', '182', '183', '184', '185', '186', '187', '188', '189',
        );
        return $tel_arr[array_rand($tel_arr)] . mt_rand(1000, 9999) . mt_rand(1000, 9999);
    }
}

if (!function_exists('hiddenMobile')) {

    /**
     * 把手机号中间四位隐藏
     *
     * @param string $mobile
     * @return void
     */
    function hiddenMobile($mobile)
    {
        return substr_replace($mobile, '****', 3, 4);
    }
}

if (!function_exists('formatNum')) {

    /**
     * 格式化数字
     *
     * @param  string $number
     */
    function formatNum($number)
    {
        if ($number >= 100000000) {
            # 判断是否超过亿
            $num = floor($number / 100000000);
            $newNum = $num . '亿+';
        } elseif ($number >= 10000) {
            # 判断是否超过万
            $num = floor($number / 10000);
            $newNum = $num . '万+';
        } elseif ($number >= 1000) {
            # 判断是否超过千
            $num = floor($number / 1000);
            $newNum = $num * 1000 . '+';
        } elseif ($number >= 100) {
            # 判断是否超过百
            $num = floor($number / 100);
            $newNum = $num * 100 . '+';
        } elseif ($number >= 10) {
            # 判断是否超过十
            $num = floor($number / 10);
            $newNum = $num * 10 . '+';
        } else {
            $newNum = $number;
        }
        return $newNum;
    }
}

if (!function_exists('existsFile')) {

    /**
     * 检测文件是不是存在
     *
     * @param string    $path
     * @param bool      $type true 远程路径、false 本地路径
     * @return bool     是否存在：true存在，false不存在
     */
    function existsFile($path, $type = true)
    {
        if ($type) {
            if (@fopen($path, 'r')) {
                return true;
            }
        } else {
            $path = public_path($path);
            if (file_exists($path)) {
                return true;
            }
        }
        return false;
    }
}

if (!function_exists('bytes')) {
    /**
     * 字节大小转换
     *
     * @param string $bytes 字节大小
     * @return string
     */
    function bytes($bytes)
    {
        $units = ['B', 'KB', 'MB', 'GB', 'TB', 'PB', 'EB', 'ZB', 'YB'];

        for ($i = 0; $bytes > 1024; $i++) {
            $bytes /= 1024;
        }

        return round($bytes, 2) . ' ' . $units[$i];
    }
}

if (!function_exists('strHidden')) {
    /**
     * 字符串截取省略
     *
     * @param  string $sourcestr
     * @param  integer $cutlength
     */
    function strHidden($sourcestr, $cutlength)
    {
        $returnstr = '';
        $i = 0;
        $n = 0;
        $str_length = strlen($sourcestr); //字符串的字节数
        while (($n < $cutlength) and ($i <= $str_length)) {
            $temp_str = substr($sourcestr, $i, 1);
            $ascnum = Ord($temp_str);   //得到字符串中第$i位字符的ascii码
            if ($ascnum >= 224)         //如果ASCII位高与224，
            {
                $returnstr = $returnstr . substr($sourcestr, $i, 3); //根据UTF-8编码规范，将3个连续的字符计为单个字符        
                $i = $i + 3;            //实际Byte计为3
                $n++;                   //字串长度计1
            } elseif ($ascnum >= 192) //如果ASCII位高与192，
            {
                $returnstr = $returnstr . substr($sourcestr, $i, 2); //根据UTF-8编码规范，将2个连续的字符计为单个字符
                $i = $i + 2;            //实际Byte计为2
                $n++;                   //字串长度计1
            } elseif ($ascnum >= 65 && $ascnum <= 90) //如果是大写字母，
            {
                $returnstr = $returnstr . substr($sourcestr, $i, 1);
                $i = $i + 1;            //实际的Byte数仍计1个
                $n++;                   //但考虑整体美观，大写字母计成一个高位字符
            } else                      //其他情况下，包括小写字母和半角标点符号，
            {
                $returnstr = $returnstr . substr($sourcestr, $i, 1);
                $i = $i + 1;            //实际的Byte数计1个
                $n = $n + 0.5;          //小写字母和半角标点等与半个高位字符宽…
            }
        }
        if ($str_length > $i) {
            $returnstr = $returnstr . "…"; //超过长度时在尾处加上省略号
        }
        return $returnstr;
    }
}

if (!function_exists('filter_Emoji')) {
    /**
     * 过滤掉emoji表情
     *
     * @param  string $str
     */
    function filter_Emoji($str)
    {
        $str = preg_replace_callback(    //执行一个正则表达式搜索并且使用一个回调进行替换
            '/./u',
            function (array $match) {
                return strlen($match[0]) >= 4 ? '' : $match[0];
            },
            $str
        );

        return $str;
    }
}

/** ------------------------- Array 数组处理 ------------------------------- */

if (!function_exists('tree')) {

    /**
     * 无限级分类 (获取 某一ID 下的所有数据)
     *
     * @param array     $arr        处理数据
     * @param integer   $parent_id  取父ID的
     * @param string    $field      父级ID字段
     * @param bool      $field      true 连同本数据返回，false 只返回下级
     * @return array    $list       返回数据
     */
    function tree($arr, $parent_id = 0, $field = 'parent_id', $type = true, &$list = [], $level = 1)
    {
        foreach ($arr as $key => $value) {
            // 计算等级
            $value['level'] = $level;
            // 数据插入
            if ($value['id'] == $parent_id) {
                $list[] = $value;
            }
            if ($value[$field] == $parent_id) {
                $list[] = $value;
                //删除已经排好的数据为了减少遍历的次数
                unset($arr[$key]);
                tree($arr, $value["id"], $field, $type, $list, $level + 1);
            }
        }
        return $list;
    }
}

if (!function_exists('_treeNode')) {

    /**
     * 无限极分类 - 指定层级
     * @param array     $data       数据
     * @param string    $parent_id  父级ID
     * @param integer   $filed      父ID字段
     * @param integer   $depth      分类的层级：0无限极
     * @param integer   $level      等级标识
     * @return array    $node       返回处理后的数据
     */
    function _treeNode($data, $parent_id = 0, $filed = 'parent_id', $depth = 3, $level = 0)
    {
        // 用于保存整理好的分类节点
        $node = [];
        $level += 1;
        // 循环所有分类
        foreach ($data as $key => &$value) {
            // 如果当前分类的父id等于要寻找的父id则写入$node数组,并寻找当前分类id下的所有子分类
            if ($parent_id == $value[$filed]) {
                $value['level'] = $level;
                $node[$key] = $value;
                unset($data[$key]);
                if ($level == $depth) {
                    $node[$key]['children'] = [];
                } else {
                    $node[$key]['children'] = _treeNode($data,  $value['id'], $filed, $depth, $level);
                }
            }
        }
        return array_values($node);
    }
}

if (!function_exists('sortData')) {

    /**
     * 对数组进行排序（二维数组）
     *
     * @param array     $data   需要排序的数据
     * @param string    $field  需要排序的字段n
     * @param string    $order  排序方式：SORT_ASC正序，SORT_DESC 倒序
     * @return array            排序后的数组
     */
    function sortData($list, $field, $order = SORT_ASC)
    {
        array_multisort(array_column($list, $field), $order, $list);
        return $list;
    }
}

if (!function_exists('sumData')) {
    /**
     * 获取一个二维数组中某个字段的总和
     *
     * @param $data
     * @return mixed
     */
    function sumData($data, $field)
    {
        return array_sum(array_column($data, $field));;
    }
}

if (!function_exists('getMin')) {
    /**
     * 获取一个二维数组中某个字段的最小值，并返回这整个对象
     *
     * @param $data     需要比较的数据
     * @param $field    需要比较的字段
     * @return mixed
     */
    function getMin($data, $field)
    {
        $quitDates  = array_column($data, $field);
        $maxItemKey = array_search(min($quitDates), $quitDates);
        return $data[$maxItemKey];
    }
}

if (!function_exists('getMax')) {
    /**
     * 获取一个二维数组中某个字段的最大值，并返回这整个对象
     *
     * @param $data     需要比较的数据
     * @param $field    需要比较的字段
     * @return mixed
     */
    function getMax($data, $field)
    {
        $quitDates  = array_column($data, $field);
        $maxItemKey = array_search(max($quitDates), $quitDates);
        return $data[$maxItemKey];
    }
}

if (!function_exists('searchData')) {
    /**
     * 搜索二维数组中某个值，并返回这整个对象
     *
     * @param $data     搜索的数据
     * @param $field    搜索的字段
     * @param $value    搜索的值
     * @return array
     */
    function searchData($data, $field, $value)
    {
        $data = array_values($data);
        $quitDates  = array_column($data, $field);
        $key = array_search($value, $quitDates);
        if ($key || $key === 0) {
            return $data[$key];
        }
        return [];
    }
}

if (!function_exists('luckDraw')) {
    /**
     * 抽奖（按比例）
     *
     * @param  array $data[
     *      key     是ID
     *      value   是比例值
     * ]
     * @param  integer $ratio
     */
    function luckDraw($data, $ratio = 100, $times = 1)
    {
        // 根据比例从大到小（这样最小比例越难中）
        arsort($data);
        foreach ($data as $k => $v) {
            $data[$k] = $v * $ratio;
        }
        $ret = '';
        $sum = array_sum($data);
        foreach ($data as $k => $v) {
            $r = mt_rand(1, $sum);
            if ($r <= $v) {
                $ret = $k;
                break;
            } else {
                if ($times <= 2) {
                    $times += 1;
                    return luckDraw($data, $ratio, $times);
                }
                $sum = max(0, $sum - $v);
            }
        }
        return $ret;
    }
}

/** ------------------------- 请求 ------------------------------- */

if (!function_exists('curlPost')) {
    /**
     * CURL POST 请求
     * @param string        $url       请求地址
     * @param array|string  $data      请求参数 array|string
     * @param array         $data      请求头['token':111]
     * @param array|json    $return    返回数据类型 true array,false 直接返回
     */
    function curlPost($url, $data, $header = [], $return = true)
    {
        $curl = curl_init();                                                // 启动一个CURL会话
        curl_setopt($curl, CURLOPT_URL, $url);                              // 要访问的地址
        curl_setopt($curl, CURLOPT_SSL_VERIFYPEER, false);                  // 对认证证书来源的检查
        curl_setopt($curl, CURLOPT_SSL_VERIFYHOST, false);                  // 从证书中检查SSL加密算法是否存在
        curl_setopt($curl, CURLOPT_USERAGENT, $_SERVER['HTTP_USER_AGENT']); // 模拟用户使用的浏览器
        if ($data != null) {
            curl_setopt($curl, CURLOPT_POST, 1);                            // 发送一个常规的Post请求
            curl_setopt($curl, CURLOPT_POSTFIELDS, $data);
        }
        if ($header) {
            curl_setopt($curl, CURLOPT_HTTPHEADER, $header);
        }
        curl_setopt($curl, CURLOPT_TIMEOUT, 300);                           // 设置超时限制防止死循环
        curl_setopt($curl, CURLOPT_HEADER, 0);                              // 显示返回的Header区域内容
        curl_setopt($curl, CURLOPT_RETURNTRANSFER, 1);                      // 获取的信息以文件流的形式返回
        $res = curl_exec($curl);                                            // 执行操作
        curl_close($curl);
        return $return ? json_decode($res, true) : $res;
    }
}

if (!function_exists('curlGet')) {

    /**
     * Get 请求
     *
     * @param string        $url    请求地址
     * @param array         $header 请求头 ['token':111]
     * @param boolean       $return 返回格式：true数组，false原格式返回
     * @return array|json
     */
    function curlGet($url, $header = [], $return = true)
    {
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        if ($header) {
            curl_setopt($ch, CURLOPT_HTTPHEADER, $header);
        }
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, FALSE);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, FALSE);
        $output = curl_exec($ch);
        curl_close($ch);
        return $return ? json_decode($output, true) : $output;
    }
}

if (!function_exists('curlXml')) {

    /**
     * xml 请求
     *
     * @param string        $url        地址
     * @param array         $header     请求头
     * @param array         $Xmldata    xml数据
     * @return array|josn
     */
    function curlXml($url, $header = [], $data)
    {
        $header[] = "Content-type: text/xml";      //定义content-type为xml,注意是数组
        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_URL, $url);
        if ($header) {
            curl_setopt($ch, CURLOPT_HTTPHEADER, $header);
        }
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $header);
        curl_setopt($ch, CURLOPT_POST, 1);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
        $response = curl_exec($ch);
        curl_close($ch);
        return $response;
    }
}
