<?php
/**
 * 朋友圈访客查询系统
 * 
 * @author    锅岛主
 * @link      https://ggdao.net
 * @wechat    IJG55555
 * @copyright Copyright (c) 2026 锅岛主 (https://ggdao.net)
 * @license   保留所有权利
 */

require_once '../config/config.php';
require_once '../includes/Database.php';
require_once '../includes/User.php';
require_once '../includes/EpayService.php';

// 创建日志目录
$log_dir = __DIR__ . '/../logs';
if (!file_exists($log_dir)) {
    mkdir($log_dir, 0777, true);
}

// 记录通知日志
$log_file = $log_dir . '/epay_notify_' . date('Y-m-d') . '.log';

// 记录完整的请求信息
$request_info = [
    'GET' => $_GET,
    'POST' => $_POST,
    'RAW' => file_get_contents('php://input')
];
file_put_contents($log_file, date('Y-m-d H:i:s') . ' - Request: ' . json_encode($request_info, JSON_UNESCAPED_UNICODE) . "\n", FILE_APPEND);

// 获取回调数据（使用GET）
$notify_data = $_GET;

// 记录解析后的通知数据
file_put_contents($log_file, date('Y-m-d H:i:s') . ' - Parsed notify data: ' . json_encode($notify_data, JSON_UNESCAPED_UNICODE) . "\n", FILE_APPEND);

if (empty($notify_data)) {
    file_put_contents($log_file, date('Y-m-d H:i:s') . " - Error: Empty notification data\n", FILE_APPEND);
    die('fail');
}

$epay = new EpayService();

// 验证通知签名
if (!$epay->verifyNotify($notify_data)) {
    file_put_contents($log_file, date('Y-m-d H:i:s') . " - Error: Invalid signature\n", FILE_APPEND);
    die('fail');
}

// 验证交易状态
if ($notify_data['trade_status'] != 'TRADE_SUCCESS') {
    file_put_contents($log_file, date('Y-m-d H:i:s') . " - Error: Invalid trade status: {$notify_data['trade_status']}\n", FILE_APPEND);
    die('fail');
}

$orderNo = $notify_data['out_trade_no'];
$tradeNo = $notify_data['trade_no'];
$money = $notify_data['money'];

$db = Database::getInstance();

try {
    // 查询订单
    $order = $db->query("SELECT * FROM vip_orders WHERE order_no = ? LIMIT 1", [$orderNo])->fetch();

    if (!$order) {
        file_put_contents($log_file, date('Y-m-d H:i:s') . " - Error: Order not found: {$orderNo}\n", FILE_APPEND);
        die('fail');
    }

    // 检查订单状态
    if ($order['status'] === 'paid') {
        file_put_contents($log_file, date('Y-m-d H:i:s') . " - Order {$orderNo} already processed\n", FILE_APPEND);
        die('success'); // 已处理的订单直接返回成功
    }

    // 检查订单金额
    if (abs($order['amount'] - $money) >= 0.01) {
        file_put_contents($log_file, date('Y-m-d H:i:s') . " - Error: Amount mismatch, order amount: {$order['amount']}, paid amount: {$money}\n", FILE_APPEND);
        die('fail');
    }

    $db->beginTransaction();
    
    try {
        // 更新订单状态
        $result = $db->query("UPDATE vip_orders SET status = 'paid', trade_no = ?, paid_at = NOW() WHERE id = ? AND status = 'pending'",
            [$tradeNo, $order['id']]);
            
        if (!$result) {
            throw new Exception("Failed to update order status");
        }
        
        // 更新用户VIP状态
        $user = new User($order['user_id']);
        if (!$user->addVipDays($order['days'])) {
            throw new Exception("Failed to add VIP days");
        }
        
        $db->commit();
        file_put_contents($log_file, date('Y-m-d H:i:s') . " - Order {$orderNo} processed successfully\n", FILE_APPEND);
        die('success');
        
    } catch (Exception $e) {
        $db->rollBack();
        file_put_contents($log_file, date('Y-m-d H:i:s') . " - Error in transaction: " . $e->getMessage() . "\n", FILE_APPEND);
        die('fail');
    }
    
} catch (Exception $e) {
    file_put_contents($log_file, date('Y-m-d H:i:s') . ' - Error processing order: ' . $e->getMessage() . "\n", FILE_APPEND);
    die('fail');
}