<?php
/**
 * 朋友圈访客查询系统
 * 
 * @author    锅岛主
 * @link      https://ggdao.net
 * @wechat    IJG55555
 * @copyright Copyright (c) 2026 锅岛主 (https://ggdao.net)
 * @license   保留所有权利
 */

require_once __DIR__ . '/Cache.php';
require_once __DIR__ . '/Config.php';

class WechatShare {
    private $cache;
    private $config;
    
    public function __construct() {
        $this->cache = new Cache();
        $this->config = Config::getInstance();
    }
    
    // 获取微信分享配置
    public function getJsConfig($url) {
        $appId = $this->config->get('wechat_appid');
        $appSecret = $this->config->get('wechat_secret');
        
        // 获取access_token
        $accessToken = $this->getAccessToken($appId, $appSecret);
        if (!$accessToken) {
            return false;
        }
        
        // 获取jsapi_ticket
        $ticket = $this->getJsApiTicket($accessToken);
        if (!$ticket) {
            return false;
        }
        
        $timestamp = time();
        $nonceStr = $this->createNonceStr();
        
        // 生成签名
        $string = "jsapi_ticket=$ticket&noncestr=$nonceStr&timestamp=$timestamp&url=$url";
        $signature = sha1($string);
        
        return [
            'appId' => $appId,
            'timestamp' => $timestamp,
            'nonceStr' => $nonceStr,
            'signature' => $signature
        ];
    }
    
    // 获取access_token
    private function getAccessToken($appId, $appSecret) {
        // 先从缓存获取
        $accessToken = $this->cache->get('wechat_access_token');
        if ($accessToken) {
            return $accessToken;
        }
        
        // 如果缓存没有，从微信服务器获取
        $url = "https://api.weixin.qq.com/cgi-bin/token?grant_type=client_credential&appid=$appId&secret=$appSecret";
        $response = file_get_contents($url);
        $json = json_decode($response, true);
        
        if (isset($json['access_token'])) {
            // 缓存access_token，有效期7000秒（比微信给的7200秒少200秒）
            $this->cache->set('wechat_access_token', $json['access_token'], 7000);
            return $json['access_token'];
        }
        
        return false;
    }
    
    // 获取jsapi_ticket
    private function getJsApiTicket($accessToken) {
        // 先从缓存获取
        $ticket = $this->cache->get('wechat_jsapi_ticket');
        if ($ticket) {
            return $ticket;
        }
        
        // 如果缓存没有，从微信服务器获取
        $url = "https://api.weixin.qq.com/cgi-bin/ticket/getticket?type=jsapi&access_token=$accessToken";
        $response = file_get_contents($url);
        $json = json_decode($response, true);
        
        if (isset($json['ticket'])) {
            // 缓存jsapi_ticket，有效期7000秒
            $this->cache->set('wechat_jsapi_ticket', $json['ticket'], 7000);
            return $json['ticket'];
        }
        
        return false;
    }
    
    // 生成随机字符串
    private function createNonceStr($length = 16) {
        $chars = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789";
        $str = "";
        for ($i = 0; $i < $length; $i++) {
            $str .= substr($chars, mt_rand(0, strlen($chars) - 1), 1);
        }
        return $str;
    }
} 