<?php
/**
 * 朋友圈访客查询系统
 * 
 * @author    锅岛主
 * @link      https://ggdao.net
 * @wechat    IJG55555
 * @copyright Copyright (c) 2026 锅岛主 (https://ggdao.net)
 * @license   保留所有权利
 */

/**
 * VIP卡密服务类
 */
class VipCardService {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance();
    }
    
    /**
     * 生成卡密
     * @param int $count 生成数量
     * @param int $days VIP天数
     * @param string|null $expireTime 过期时间
     * @return array 生成结果
     */
    public function generateCards($count, $days, $expireTime = null) {
        $batchNo = date('YmdHis') . mt_rand(1000, 9999);
        $cards = [];
        $success = 0;
        
        try {
            $this->db->beginTransaction();
            
            for ($i = 0; $i < $count; $i++) {
                $cardNo = $this->generateCardNo();
                
                $sql = "INSERT INTO vip_cards (card_no, days, batch_no, status, expire_time) VALUES (?, ?, ?, 'unused', ?)";
                $result = $this->db->query($sql, [$cardNo, $days, $batchNo, $expireTime]);
                
                if ($result) {
                    $cards[] = $cardNo;
                    $success++;
                }
            }
            
            $this->db->commit();
            return [
                'code' => 0,
                'msg' => '成功生成' . $success . '张卡密',
                'data' => [
                    'batch_no' => $batchNo,
                    'cards' => $cards
                ]
            ];
        } catch (Exception $e) {
            $this->db->rollBack();
            return ['code' => 1, 'msg' => '生成卡密失败：' . $e->getMessage()];
        }
    }
    
    /**
     * 验证并使用卡密
     * @param string $cardNo 卡密号码
     * @param int $userId 用户ID
     * @return array 使用结果
     */
    public function useCard($cardNo, $userId) {
        try {
            $this->db->beginTransaction();
            
            // 查询卡密
            $sql = "SELECT * FROM vip_cards WHERE card_no = ? LIMIT 1";
            $card = $this->db->query($sql, [$cardNo])->fetch();
            
            if (!$card) {
                throw new Exception('卡密不存在');
            }
            
            if ($card['status'] !== 'unused') {
                throw new Exception('卡密已被使用或已作废');
            }
            
            if ($card['expire_time'] && strtotime($card['expire_time']) < time()) {
                throw new Exception('卡密已过期');
            }
            
            // 更新卡密状态
            $sql = "UPDATE vip_cards SET 
                    status = 'used',
                    used_user_id = ?,
                    used_time = NOW()
                    WHERE id = ? AND status = 'unused'";
            $result = $this->db->query($sql, [$userId, $card['id']]);
            
            if (!$result) {
                throw new Exception('卡密使用失败');
            }
            
            // 更新用户VIP状态
            $user = new User();
            if (!$user->addVipDays($card['days'])) {
                throw new Exception('更新会员状态失败');
            }
            
            $this->db->commit();
            return ['code' => 0, 'msg' => '卡密使用成功'];
            
        } catch (Exception $e) {
            $this->db->rollBack();
            return ['code' => 1, 'msg' => $e->getMessage()];
        }
    }
    
    /**
     * 作废卡密
     * @param string|array $cardNos 卡密号码
     * @return array 操作结果
     */
    public function invalidateCards($cardNos) {
        if (!is_array($cardNos)) {
            $cardNos = [$cardNos];
        }
        
        try {
            $placeholders = str_repeat('?,', count($cardNos) - 1) . '?';
            $sql = "UPDATE vip_cards SET status = 'invalid' WHERE card_no IN ($placeholders) AND status = 'unused'";
            $result = $this->db->query($sql, $cardNos);
            
            return [
                'code' => 0,
                'msg' => '成功作废' . $result->rowCount() . '张卡密'
            ];
        } catch (Exception $e) {
            return ['code' => 1, 'msg' => '作废卡密失败：' . $e->getMessage()];
        }
    }
    
    /**
     * 获取卡密统计信息
     * @return array
     */
    public function getCardStats() {
        $sql = "SELECT 
                COUNT(*) as total,
                SUM(CASE WHEN status = 'unused' THEN 1 ELSE 0 END) as unused,
                SUM(CASE WHEN status = 'used' THEN 1 ELSE 0 END) as used,
                SUM(CASE WHEN status = 'invalid' THEN 1 ELSE 0 END) as invalid
                FROM vip_cards";
        
        return $this->db->query($sql)->fetch();
    }
    
    /**
     * 生成卡密号码
     * @return string
     */
    private function generateCardNo() {
        $chars = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789';
        $cardNo = '';
        
        // 生成16位卡密，每4位用-分隔
        for ($i = 0; $i < 16; $i++) {
            if ($i > 0 && $i % 4 === 0) {
                $cardNo .= '-';
            }
            $cardNo .= $chars[mt_rand(0, strlen($chars) - 1)];
        }
        
        return $cardNo;
    }
} 