<?php
/**
 * 朋友圈访客查询系统
 * 
 * @author    锅岛主
 * @link      https://ggdao.net
 * @wechat    IJG55555
 * @copyright Copyright (c) 2026 锅岛主 (https://ggdao.net)
 * @license   保留所有权利
 */

require_once __DIR__ . '/Config.php';

class User {
    private $db;
    private $config;
    private $userId;
    private $debug = true;  // 添加调试模式属性

    public function __construct($userId = null) {
        $this->db = Database::getInstance();
        $this->config = Config::getInstance();
        $this->userId = $userId ?? ($_SESSION['user']['id'] ?? null);
    }

    public function handleWechatCallback($code) {
        $url = sprintf(
            "https://api.weixin.qq.com/sns/oauth2/access_token?appid=%s&secret=%s&code=%s&grant_type=authorization_code",
            $this->config->get('wechat_appid'),
            $this->config->get('wechat_secret'),
            $code
        );
        
        $response = file_get_contents($url);
        $data = json_decode($response, true);
        
        if (isset($data['openid'])) {
            $accessToken = $data['access_token'];
            $openid = $data['openid'];
            
            // 获取用户信息
            $userInfoUrl = sprintf(
                "https://api.weixin.qq.com/sns/userinfo?access_token=%s&openid=%s&lang=zh_CN",
                $accessToken,
                $openid
            );
            
            $userInfo = json_decode(file_get_contents($userInfoUrl), true);
            
            if ($userInfo) {
                // 查询用户是否存在
                $stmt = $this->db->query("SELECT * FROM users WHERE openid = ?", [$openid]);
                $user = $stmt->fetch();
                
                if (!$user) {
                    // 获取来源信息
                    $source = 'direct';
                    $shareFromPostId = null;
                    $shareFromUserId = null;
                    
                    // 如果是通过分享链接进来的
                    if (isset($_SESSION['share_from'])) {
                        $source = 'share';
                        $shareFromPostId = $_SESSION['share_from']['post_id'] ?? null;
                        $shareFromUserId = $_SESSION['share_from']['user_id'] ?? null;
                        // 清除分享来源session
                        unset($_SESSION['share_from']);
                    }
                    
                    // 新用户，插入数据库
                    $data = [
                        'openid' => $openid,
                        'nickname' => $userInfo['nickname'],
                        'avatar' => $userInfo['headimgurl'],
                        'source' => $source,
                        'share_from_post_id' => $shareFromPostId,
                        'share_from_user_id' => $shareFromUserId,
                        'created_at' => date('Y-m-d H:i:s'),
                        'last_login' => date('Y-m-d H:i:s')
                    ];
                    $this->db->insert('users', $data);
                    $userId = $this->db->getConnection()->lastInsertId();
                    
                    // 获取新插入的用户信息
                    $stmt = $this->db->query("SELECT * FROM users WHERE id = ?", [$userId]);
                    $user = $stmt->fetch();
                } else {
                    $userId = $user['id'];
                    // 更新最后登录时间和用户信息
                    $this->db->update('users', 
                        [
                            'last_login' => date('Y-m-d H:i:s'),
                            'nickname' => $userInfo['nickname'],
                            'avatar' => $userInfo['headimgurl']
                        ],
                        'id = ?',
                        [$userId]
                    );
                }
                
                // 设置会话
                $_SESSION['user'] = [
                    'id' => $userId,
                    'openid' => $openid,
                    'nickname' => $userInfo['nickname'],
                    'avatar' => $userInfo['headimgurl'],
                    'role' => $user['role']
                ];
                
                return true;
            }
        }
        
        return false;
    }

    // 记录用户来源
    public function recordSource($userId, $postId = null, $shareUserId = null) {
        if ($postId && $shareUserId) {
            $sql = "UPDATE users SET 
                    source = 'share',
                    share_from_post_id = ?,
                    share_from_user_id = ?
                    WHERE id = ?";
            $this->db->query($sql, [$postId, $shareUserId, $userId]);
        }
    }

    // 获取用户增长统计
    public function getGrowthStats($date = null) {
        if (!$date) {
            $date = date('Y-m-d');
        }
        
        $stats = [
            'total_new_users' => 0,
            'direct_users' => 0,
            'share_users' => 0,
            'share_active_users' => 0
        ];
        
        // 获取今日新增用户总数
        $sql = "SELECT 
                COUNT(*) as total,
                SUM(CASE WHEN source = 'direct' THEN 1 ELSE 0 END) as direct_count,
                SUM(CASE WHEN source = 'share' THEN 1 ELSE 0 END) as share_count
                FROM users 
                WHERE DATE(created_at) = ?";
        $result = $this->db->query($sql, [$date])->fetch();
        
        $stats['total_new_users'] = $result['total'];
        $stats['direct_users'] = $result['direct_count'];
        $stats['share_users'] = $result['share_count'];
        
        // 获取通过分享来的用户中，有多少发布了动态
        $sql = "SELECT COUNT(DISTINCT u.id) as active_count
                FROM users u
                JOIN posts p ON u.id = p.user_id
                WHERE u.source = 'share'
                AND DATE(u.created_at) = ?
                AND DATE(p.created_at) = ?";
        $result = $this->db->query($sql, [$date, $date])->fetch();
        $stats['share_active_users'] = $result['active_count'];
        
        return $stats;
    }

    /**
     * 检查用户是否是VIP会员
     * @return bool
     */
    public function isVip() {
        if (!isset($_SESSION['user'])) {
            return false;
        }
        
        $userId = $_SESSION['user']['id'];
        $sql = "SELECT vip_level, vip_expire_time FROM users WHERE id = ?";
        $result = $this->db->query($sql, [$userId])->fetch();
        
        if (!$result) {
            return false;
        }
        
        // 检查是否是VIP且未过期
        if ($result['vip_level'] == 1 && $result['vip_expire_time'] > date('Y-m-d H:i:s')) {
            return true;
        }
        
        return false;
    }
    
    /**
     * 获取VIP过期时间
     * @return string|null
     */
    public function getVipExpireTime() {
        if (!isset($_SESSION['user'])) {
            return null;
        }
        
        $userId = $_SESSION['user']['id'];
        $sql = "SELECT vip_expire_time FROM users WHERE id = ?";
        $result = $this->db->query($sql, [$userId])->fetch();
        
        return $result ? $result['vip_expire_time'] : null;
    }
    
    /**
     * 创建VIP订单
     * @param float $amount 订单金额
     * @param int $days VIP天数
     * @return string|false 成功返回订单号，失败返回false
     */
    public function createVipOrder($amount, $days) {
        if (!isset($_SESSION['user'])) {
            return false;
        }
        
        $userId = $_SESSION['user']['id'];
        $orderNo = date('YmdHis') . rand(1000, 9999);
        
        $sql = "INSERT INTO vip_orders (user_id, order_no, amount, days, status) VALUES (?, ?, ?, ?, 'pending')";
        $stmt = $this->db->prepare($sql);
        $stmt->bind_param('isdi', $userId, $orderNo, $amount, $days);
        
        return $stmt->execute() ? $orderNo : false;
    }
    
    /**
     * 激活VIP会员
     * @param string $orderNo 订单号
     * @return bool
     */
    public function activateVip($orderNo) {
        if (!isset($_SESSION['user'])) {
            return false;
        }
        
        $userId = $_SESSION['user']['id'];
        
        // 开启事务
        $this->db->begin_transaction();
        
        try {
            // 查询订单
            $sql = "SELECT * FROM vip_orders WHERE order_no = ? AND user_id = ? AND status = 'pending'";
            $stmt = $this->db->prepare($sql);
            $stmt->bind_param('si', $orderNo, $userId);
            $stmt->execute();
            $order = $stmt->get_result()->fetch_assoc();
            
            if (!$order) {
                throw new Exception('订单不存在或状态异常');
            }
            
            // 更新订单状态
            $sql = "UPDATE vip_orders SET status = 'paid', paid_at = NOW() WHERE id = ?";
            $stmt = $this->db->prepare($sql);
            $stmt->bind_param('i', $order['id']);
            $stmt->execute();
            
            // 更新用户VIP状态
            $expireTime = date('Y-m-d H:i:s', strtotime("+{$order['days']} days"));
            $sql = "UPDATE users SET vip_level = 1, vip_expire_time = ? WHERE id = ?";
            $stmt = $this->db->prepare($sql);
            $stmt->bind_param('si', $expireTime, $userId);
            $stmt->execute();
            
            // 提交事务
            $this->db->commit();
            return true;
            
        } catch (Exception $e) {
            // 回滚事务
            $this->db->rollback();
            return false;
        }
    }

    /**
     * 添加VIP天数
     * @param int $days 要添加的天数
     * @return bool
     */
    public function addVipDays($days) {
        if (!$this->userId) {
            error_log("Error: No user ID provided in addVipDays");
            return false;
        }

        try {
            // 获取用户当前VIP状态
            $sql = "SELECT vip_level, vip_expire_time FROM users WHERE id = ?";
            $result = $this->db->query($sql, [$this->userId])->fetch();
            
            if (!$result) {
                error_log("Error: User not found in addVipDays for user_id: " . $this->userId);
                return false;
            }
            
            // 计算新的过期时间
            $now = time();
            $currentExpireTime = $result['vip_expire_time'] 
                ? strtotime($result['vip_expire_time']) 
                : $now;
            
            // 如果已过期，从当前时间开始计算
            if ($currentExpireTime < $now) {
                $currentExpireTime = $now;
            }
            
            // 计算新的过期时间
            $newExpireTime = date('Y-m-d H:i:s', $currentExpireTime + ($days * 86400));
            
            // 更新用户VIP状态
            $sql = "UPDATE users SET vip_level = 1, vip_expire_time = ? WHERE id = ?";
            $result = $this->db->query($sql, [$newExpireTime, $this->userId]);
            
            if (!$result) {
                error_log("Error: Failed to update VIP status for user_id: " . $this->userId);
                return false;
            }
            
            error_log("Success: VIP days added for user_id: " . $this->userId . ", new expire time: " . $newExpireTime);
            return true;
            
        } catch (Exception $e) {
            error_log("Exception in addVipDays: " . $e->getMessage());
            return false;
        }
    }
} 