<?php
/**
 * 朋友圈访客查询系统
 * 
 * @author    锅岛主
 * @link      https://ggdao.net
 * @wechat    IJG55555
 * @copyright Copyright (c) 2026 锅岛主 (https://ggdao.net)
 * @license   保留所有权利
 */

class Post {
    private $db;

    public function __construct() {
        $this->db = Database::getInstance();
    }

    // 创建新动态
    public function create($userId, $content, $image = null) {
        try {
            if (DEBUG_MODE) {
                error_log("Creating post with image: " . $image);
            }
            
            $this->db->beginTransaction();
            
            $shareCode = $this->generateShareCode();
            $data = [
                'user_id' => $userId,
                'content' => $content,
                'image' => $image,
                'share_code' => $shareCode,
                'created_at' => date('Y-m-d H:i:s')
            ];
            
            if (DEBUG_MODE) {
                error_log("Post data: " . print_r($data, true));
            }
            
            $stmt = $this->db->insert('posts', $data);
            $postId = $this->db->getConnection()->lastInsertId();
            
            $this->db->commit();
            return $postId;
            
        } catch (Exception $e) {
            $this->db->rollBack();
            if (DEBUG_MODE) {
                error_log("Error creating post: " . $e->getMessage());
            }
            throw $e;
        }
    }

    // 获取用户的所有动态
    public function getUserPosts($userId, $page = 1, $limit = 10) {
        $offset = ($page - 1) * $limit;
        $sql = "SELECT p.*, u.nickname, u.avatar,
                (SELECT COUNT(DISTINCT visitor_id) FROM visitors WHERE post_id = p.id) as visitor_count,
                p.view_count as total_views
                FROM posts p 
                JOIN users u ON p.user_id = u.id 
                WHERE p.user_id = ? AND p.status = 'active'
                ORDER BY p.created_at DESC
                LIMIT ? OFFSET ?";
        return $this->db->query($sql, [$userId, $limit, $offset])->fetchAll();
    }

    // 获取单条动态详情
    public function getPost($postId) {
        $sql = "
            SELECT 
                p.*,
                u.nickname,
                u.avatar,
                COUNT(DISTINCT v.id) as view_count,
                COUNT(DISTINCT v.visitor_id) as visitor_count
            FROM posts p
            JOIN users u ON p.user_id = u.id
            LEFT JOIN visitors v ON p.id = v.post_id
            WHERE p.id = ? AND p.status = 'active'
            GROUP BY p.id, p.user_id, p.content, p.image, p.share_code, p.created_at, p.status, u.nickname, u.avatar
        ";
        
        $post = $this->db->query($sql, [$postId])->fetch();
        
        if ($post && DEBUG_MODE) {
            error_log("Post ID: " . $postId);
            error_log("Image path: " . $post['image']);
        }
        
        return $post;
    }

    // 通过分享码获取动态
    public function getPostByShareCode($shareCode) {
        $sql = "SELECT p.*, u.nickname, u.avatar,
                (SELECT COUNT(DISTINCT visitor_id) FROM visitors WHERE post_id = p.id) as visitor_count,
                p.view_count as total_views
                FROM posts p 
                JOIN users u ON p.user_id = u.id 
                WHERE p.share_code = ? AND p.status = 'active'";
        return $this->db->query($sql, [$shareCode])->fetch();
    }

    // 记录访客
    public function recordVisitor($postId, $visitorId) {
        try {
            $this->db->beginTransaction();
            
            // 检查是否是新访客
            $sql = "SELECT COUNT(*) as count FROM visitors WHERE post_id = ? AND visitor_id = ?";
            $result = $this->db->query($sql, [$postId, $visitorId])->fetch();
            
            // 只有新访客才增加浏览次数
            if ($result['count'] == 0) {
                $sql = "UPDATE posts SET view_count = view_count + 1 WHERE id = ?";
                $this->db->query($sql, [$postId]);
            }
            
            // 记录访客
            $data = [
                'post_id' => $postId,
                'visitor_id' => $visitorId,
                'ip_address' => $_SERVER['REMOTE_ADDR'],
                'user_agent' => $_SERVER['HTTP_USER_AGENT'],
                'visited_at' => date('Y-m-d H:i:s')
            ];
            
            $this->db->insert('visitors', $data);
            
            $this->db->commit();
            return true;
            
        } catch (Exception $e) {
            $this->db->rollBack();
            throw $e;
        }
    }

    // 获取动态的访客列表
    public function getVisitors($postId) {
        $sql = "SELECT v.*, u.nickname, u.avatar
                FROM visitors v
                JOIN users u ON v.visitor_id = u.id
                INNER JOIN (
                    SELECT visitor_id, MAX(visited_at) as last_visit
                    FROM visitors
                    WHERE post_id = ?
                    GROUP BY visitor_id
                ) last_visits ON v.visitor_id = last_visits.visitor_id 
                    AND v.visited_at = last_visits.last_visit
                WHERE v.post_id = ?
                ORDER BY v.visited_at DESC";
        return $this->db->query($sql, [$postId, $postId])->fetchAll();
    }

    // 删除动态
    public function delete($postId, $userId) {
        $sql = "UPDATE posts SET status = 'deleted' WHERE id = ? AND user_id = ?";
        return $this->db->query($sql, [$postId, $userId])->rowCount() > 0;
    }

    // 生成分享码
    private function generateShareCode($length = 8) {
        $chars = 'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789';
        $code = '';
        
        do {
            $code = '';
            for ($i = 0; $i < $length; $i++) {
                $code .= $chars[rand(0, strlen($chars) - 1)];
            }
            
            // 检查是否已存在
            $sql = "SELECT COUNT(*) as count FROM posts WHERE share_code = ?";
            $result = $this->db->query($sql, [$code])->fetch();
            
        } while ($result['count'] > 0);
        
        return $code;
    }

    public function recordVisit($postId, $visitorId) {
        // 检查是否已经访问过
        $sql = "SELECT id FROM visitors WHERE post_id = ? AND visitor_id = ? AND DATE(visited_at) = CURDATE()";
        $result = $this->db->query($sql, [$postId, $visitorId])->fetch();
        
        if (!$result) {
            // 添加访问记录
            $sql = "INSERT INTO visitors (post_id, visitor_id, visited_at) VALUES (?, ?, NOW())";
            $this->db->query($sql, [$postId, $visitorId]);
            
            // 更新动态的访问次数
            $sql = "UPDATE posts SET view_count = view_count + 1 WHERE id = ?";
            $this->db->query($sql, [$postId]);
        }
    }

    public function getUserStats($userId) {
        // 获取今日开始时间
        $todayStart = date('Y-m-d 00:00:00');
        
        // 获取今日浏览次数
        $sql = "SELECT COUNT(*) as count FROM visitors 
                WHERE post_id IN (SELECT id FROM posts WHERE user_id = ? AND status = 'active')
                AND visited_at >= ?";
        $todayViews = $this->db->query($sql, [$userId, $todayStart])->fetch()['count'];
        
        // 获取今日访客数
        $sql = "SELECT COUNT(DISTINCT visitor_id) as count FROM visitors 
                WHERE post_id IN (SELECT id FROM posts WHERE user_id = ? AND status = 'active')
                AND visited_at >= ?";
        $todayVisitors = $this->db->query($sql, [$userId, $todayStart])->fetch()['count'];
        
        // 获取总访客数
        $sql = "SELECT COUNT(DISTINCT visitor_id) as count FROM visitors 
                WHERE post_id IN (SELECT id FROM posts WHERE user_id = ? AND status = 'active')";
        $totalVisitors = $this->db->query($sql, [$userId])->fetch()['count'];
        
        return [
            'today_views' => $todayViews,
            'today_visitors' => $todayVisitors,
            'total_visitors' => $totalVisitors
        ];
    }

    public function getStats() {
        $stats = [
            'total_users' => 0,
            'today_visits' => 0,
            'recent_users' => []
        ];
        
        // 获取总用户数
        $result = $this->db->query("SELECT COUNT(*) as count FROM users")->fetch();
        $stats['total_users'] = $result['count'];
        
        // 获取今日访问量
        $today = date('Y-m-d');
        $result = $this->db->query(
            "SELECT COUNT(*) as count FROM visitors WHERE DATE(visited_at) = ?", 
            [$today]
        )->fetch();
        $stats['today_visits'] = $result['count'];
        
        // 获取最近的用户（最多10个）
        $sql = "
            SELECT DISTINCT 
                u.id,
                u.nickname,
                u.avatar,
                MAX(v.visited_at) as visited_at
            FROM visitors v
            JOIN users u ON v.visitor_id = u.id
            WHERE v.visited_at >= DATE_SUB(NOW(), INTERVAL 1 DAY)
            GROUP BY u.id, u.nickname, u.avatar
            ORDER BY visited_at DESC
            LIMIT 10
        ";
        $stats['recent_users'] = $this->db->query($sql)->fetchAll();
        
        return $stats;
    }

    // 记录动态分享状态
    public function recordShare($postId) {
        $sql = "UPDATE posts SET 
                share_status = 'shared',
                first_share_time = CASE 
                    WHEN first_share_time IS NULL THEN NOW()
                    ELSE first_share_time
                END
                WHERE id = ?";
        return $this->db->query($sql, [$postId]);
    }

    // 获取动态分享统计
    public function getShareStats($date) {
        $stats = [
            'total_posts' => 0,      // 新增动态数
            'shared_posts' => 0,     // 被分享的动态数
            'effective_shares' => 0,  // 有效分享数(被访问)
        ];
        
        // 获取指定日期的新增动态数
        $sql = "SELECT COUNT(*) as count FROM posts 
                WHERE DATE(created_at) = ? AND status = 'active'";
        $result = $this->db->query($sql, [$date])->fetch();
        $stats['total_posts'] = $result['count'];
        
        // 获取被分享且有访问的动态数
        $sql = "SELECT 
                    COUNT(DISTINCT p.id) as shared_posts,
                    COUNT(DISTINCT CASE WHEN v.visitor_id != p.user_id THEN p.id END) as effective_shares
                FROM posts p
                LEFT JOIN visitors v ON p.id = v.post_id
                WHERE DATE(p.created_at) = ? 
                AND p.status = 'active'";
        $result = $this->db->query($sql, [$date])->fetch();
        
        $stats['shared_posts'] = $result['shared_posts'];
        $stats['effective_shares'] = $result['effective_shares'];
        
        return $stats;
    }
} 