<?php
/**
 * 朋友圈访客查询系统
 * 
 * @author    锅岛主
 * @link      https://ggdao.net
 * @wechat    IJG55555
 * @copyright Copyright (c) 2026 锅岛主 (https://ggdao.net)
 * @license   保留所有权利
 */

/**
 * 易支付服务类
 */
class EpayService {
    private $config = [];
    private $debug = true;  // 开启调试模式
    
    public function __construct() {
        $this->loadConfig();
    }
    
    /**
     * 加载易支付配置
     */
    private function loadConfig() {
        $db = Database::getInstance();
        $configs = $db->query("SELECT `key`, `value` FROM system_config WHERE `key` IN ('epay_url', 'epay_pid', 'epay_key', 'epay_notify_url', 'epay_return_url', 'site_url')")->fetchAll(PDO::FETCH_KEY_PAIR);
        
        // 获取站点URL，如果未配置则使用当前域名
        $siteUrl = !empty($configs['site_url']) ? rtrim($configs['site_url'], '/') : '';
        if (empty($siteUrl)) {
            $siteUrl = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? "https" : "http") . "://$_SERVER[HTTP_HOST]";
        }
        
        $this->config = [
            'api_url' => rtrim($configs['epay_url'] ?? '', '/'),
            'pid' => $configs['epay_pid'] ?? '',
            'key' => $configs['epay_key'] ?? '',
            'notify_url' => str_replace('{site_url}', $siteUrl, $configs['epay_notify_url'] ?? $siteUrl . '/vip/notify.php'),
            'return_url' => str_replace('{site_url}', $siteUrl, $configs['epay_return_url'] ?? $siteUrl . '/vip/return.php')
        ];
    }
    
    /**
     * 创建支付订单
     * @param string $order_no 订单号
     * @param float $amount 金额
     * @param string $subject 商品名称
     * @return string 支付跳转链接
     */
    public function createOrder($order_no, $amount, $subject) {
        $params = [
            'pid' => $this->config['pid'],
            'type' => 'wxpay',
            'out_trade_no' => $order_no,
            'notify_url' => $this->config['notify_url'],
            'return_url' => $this->config['return_url'],
            'name' => $subject,
            'money' => number_format($amount, 2, '.', ''),
            'sitename' => '朋友圈访客查询系统'
        ];
        
        // 生成签名
        $params['sign'] = $this->getSign($params);
        $params['sign_type'] = 'MD5';
        
        if ($this->debug) {
            error_log("[EpayService] Create Order Params: " . json_encode($params, JSON_UNESCAPED_UNICODE));
        }
        
        // 返回支付链接
        return $this->config['api_url'] . '/submit.php?' . http_build_query($params);
    }
    
    /**
     * 验证异步通知
     * @param array $params 通知参数
     * @return bool 验证结果
     */
    public function verifyNotify($params) {
        if ($this->debug) {
            error_log("[EpayService] Verify Notify Params: " . json_encode($params, JSON_UNESCAPED_UNICODE));
        }
        
        if (empty($params)) {
            error_log("[EpayService] Error: Empty notification data");
            return false;
        }
        
        if ($params['pid'] != $this->config['pid']) {
            error_log("[EpayService] Error: PID mismatch, expected {$this->config['pid']}, got {$params['pid']}");
            return false;
        }
        
        if (empty($params['sign'])) {
            error_log("[EpayService] Error: Sign is empty");
            return false;
        }
        
        $sign = $params['sign'];
        
        // 生成签名
        $signStr = $this->getSign($params);
        
        if ($this->debug) {
            error_log("[EpayService] Original sign: " . $sign);
            error_log("[EpayService] Generated sign: " . $signStr);
        }
        
        return $sign === $signStr;
    }
    
    /**
     * 生成签名
     * @param array $params 参数
     * @return string 签名
     */
    private function getSign($params) {
        ksort($params);
        reset($params);
        $signstr = '';
        
        foreach($params as $k => $v) {
            if($k != "sign" && $k != "sign_type" && $v !== '') {
                $signstr .= $k.'='.$v.'&';
            }
        }
        $signstr = substr($signstr, 0, -1);
        $signstr .= $this->config['key'];
        
        if ($this->debug) {
            error_log("[EpayService] String to sign: " . $signstr);
        }
        
        return md5($signstr);
    }
    
    /**
     * 测试配置是否正确
     * @return array 测试结果
     */
    public function testConfig() {
        if(empty($this->config['api_url']) || empty($this->config['pid']) || empty($this->config['key'])) {
            return ['code' => 1, 'msg' => '请先完善易支付配置信息'];
        }
        
        // 测试接口连通性
        $test_url = $this->config['api_url'] . '/api.php?act=query&pid=' . $this->config['pid'];
        $ch = curl_init($test_url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if($http_code != 200) {
            return ['code' => 1, 'msg' => '易支付接口地址无法访问'];
        }
        
        $result = json_decode($response, true);
        if(empty($result)) {
            return ['code' => 1, 'msg' => '易支付接口返回数据异常'];
        }
        
        return ['code' => 0, 'msg' => '配置信息正确'];
    }
}