<?php
/**
 * 朋友圈访客查询系统
 * 
 * @author    锅岛主
 * @link      https://ggdao.net
 * @wechat    IJG55555
 * @copyright Copyright (c) 2026 锅岛主 (https://ggdao.net)
 * @license   保留所有权利
 */

class Config {
    private $db;
    private static $instance = null;
    private $config = [];
    
    // 默认配置
    private $defaults = [
        'wechat_appid' => '',  // 留空，需要在后台配置
        'wechat_secret' => '', // 留空，需要在后台配置
        'wechat_oauth_api' => '', // 留空则使用官方API，填写则使用第三方回调服务
        'site_url' => '',      // 留空，需要在后台配置
        'site_name' => '朋友圈访客',
        'share_default_image' => '/assets/images/share-logo.png'
    ];

    private function __construct() {
        $this->db = Database::getInstance();
        $this->loadConfig();
    }

    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    // 加载所有配置
    private function loadConfig() {
        try {
            $sql = "SELECT * FROM system_config";
            $configs = $this->db->query($sql)->fetchAll();
            foreach ($configs as $config) {
                $this->config[$config['key']] = $config['value'];
            }
        } catch (Exception $e) {
            // 如果数据库表不存在或查询失败，使用默认配置
            $this->config = $this->defaults;
        }
    }

    // 获取配置值
    public function get($key, $default = null) {
        // 优先从数据库配置获取
        if (isset($this->config[$key])) {
            return $this->config[$key];
        }
        // 其次从默认配置获取
        if (isset($this->defaults[$key])) {
            return $this->defaults[$key];
        }
        // 最后返��传入的默认值
        return $default;
    }

    // 设置配置值
    public function set($key, $value) {
        try {
            $this->db->beginTransaction();
            
            $sql = "INSERT INTO system_config (key, value) 
                   VALUES (?, ?) 
                   ON DUPLICATE KEY UPDATE value = ?";
            $this->db->query($sql, [$key, $value, $value]);
            
            $this->config[$key] = $value;
            
            $this->db->commit();
            return true;
        } catch (Exception $e) {
            $this->db->rollBack();
            return false;
        }
    }

    // 获取所有配置
    public function getAll() {
        try {
            $sql = "SELECT * FROM system_config ORDER BY id ASC";
            return $this->db->query($sql)->fetchAll();
        } catch (Exception $e) {
            // 如果数据库表不存在，返回默认配置的格式
            $configs = [];
            foreach ($this->defaults as $key => $value) {
                $configs[] = [
                    'key' => $key,
                    'value' => $value,
                    'description' => $this->getDescription($key),
                    'type' => 'text'
                ];
            }
            return $configs;
        }
    }

    // 获取配置项描述
    private function getDescription($key) {
        $descriptions = [
            'wechat_appid' => '微信公众号AppID',
            'wechat_secret' => '微信公众号Secret',
            'wechat_oauth_api' => '公众号无限回调登录接口',
            'site_url' => '站点URL',
            'site_name' => '站点名称',
            'share_default_image' => '默认分享图片'
        ];
        return isset($descriptions[$key]) ? $descriptions[$key] : $key;
    }

    // 批量更新配置
    public function batchUpdate($configs) {
        try {
            $this->db->beginTransaction();
            
            foreach ($configs as $key => $value) {
                $sql = "UPDATE system_config SET value = ? WHERE `key` = ?";
                $this->db->query($sql, [$value, $key]);
                $this->config[$key] = $value;
            }
            
            $this->db->commit();
            return true;
        } catch (Exception $e) {
            $this->db->rollBack();
            return false;
        }
    }
} 