<?php
/**
 * 朋友圈访客查询系统
 * 
 * @author    锅岛主
 * @link      https://ggdao.net
 * @wechat    IJG55555
 * @copyright Copyright (c) 2026 锅岛主 (https://ggdao.net)
 * @license   保留所有权利
 */

$pageTitle = '用户管理';
$pageIcon = 'fas fa-users';

require_once '../includes/Admin.php';
$admin = new Admin();

// 获取当前页码
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$page = max(1, $page);
$limit = 10;

// 获取用户列表和总数
$users = $admin->getAllUsers($page, $limit);
$totalUsers = $admin->getUserCount();
$totalPages = ceil($totalUsers / $limit);

require_once 'layout/header.php';
?>

<!-- 用户列表 -->
<div class="table-card">
    <div class="card-header d-flex justify-content-between align-items-center">
        <h5 class="card-title">用户列表</h5>
        <div class="d-flex align-items-center">
            <span class="text-muted me-3">
                共 <?php echo number_format($totalUsers); ?> 个用户
            </span>
            <div class="input-group" style="width: 300px;">
                <input type="text" class="form-control" id="searchInput" placeholder="搜索用户...">
                <button class="btn btn-outline-secondary" type="button" id="searchBtn">
                    <i class="fas fa-search"></i>
                </button>
            </div>
        </div>
    </div>
    <div class="table-responsive">
        <table class="table table-hover">
            <thead>
                <tr>
                    <th>ID</th>
                    <th>头像</th>
                    <th>昵称</th>
                    <th>来源</th>
                    <th>动态数</th>
                    <th>有效分享</th>
                    <th>带来用户</th>
                    <th>注册时间</th>
                    <th>最后登录</th>
                    <th>操作</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($users as $user): ?>
                <tr>
                    <td><?php echo $user['id']; ?></td>
                    <td>
                        <img src="<?php echo htmlspecialchars($user['avatar']); ?>" 
                             class="rounded-circle" 
                             width="40" height="40" 
                             alt="<?php echo htmlspecialchars($user['nickname']); ?>">
                    </td>
                    <td><?php echo htmlspecialchars($user['nickname']); ?></td>
                    <td>
                        <?php if ($user['source'] == 'share'): ?>
                            <span class="badge bg-success">裂变</span>
                            <?php if ($user['share_from_user_id']): ?>
                                <small class="text-muted">
                                    来自用户ID: <?php echo $user['share_from_user_id']; ?>
                                </small>
                            <?php endif; ?>
                        <?php else: ?>
                            <span class="badge bg-secondary">自然</span>
                        <?php endif; ?>
                    </td>
                    <td>
                        <?php if ($user['total_posts'] > 0): ?>
                            <span class="badge bg-info"><?php echo $user['total_posts']; ?></span>
                        <?php else: ?>
                            <span class="text-muted">-</span>
                        <?php endif; ?>
                    </td>
                    <td>
                        <?php if ($user['effective_shares'] > 0): ?>
                            <span class="badge bg-success"><?php echo $user['effective_shares']; ?></span>
                        <?php else: ?>
                            <span class="text-muted">-</span>
                        <?php endif; ?>
                    </td>
                    <td>
                        <?php if ($user['referral_count'] > 0): ?>
                            <span class="badge bg-primary"><?php echo $user['referral_count']; ?>人</span>
                        <?php else: ?>
                            <span class="text-muted">-</span>
                        <?php endif; ?>
                    </td>
                    <td><?php echo date('Y-m-d H:i', strtotime($user['created_at'])); ?></td>
                    <td>
                        <?php
                        if ($user['last_login']) {
                            echo date('Y-m-d H:i', strtotime($user['last_login']));
                        } else {
                            echo '<span class="text-muted">从未登录</span>';
                        }
                        ?>
                    </td>
                    <td>
                        <?php if ($user['role'] !== 'admin'): ?>
                        <button type="button" 
                                class="btn btn-sm btn-danger delete-user" 
                                data-id="<?php echo $user['id']; ?>"
                                data-name="<?php echo htmlspecialchars($user['nickname']); ?>">
                            <i class="fas fa-trash-alt"></i>
                        </button>
                        <?php endif; ?>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>

    <!-- 分页 -->
    <?php if ($totalPages > 1): ?>
    <div class="card-footer">
        <nav>
            <ul class="pagination justify-content-center mb-0">
                <?php if ($page > 1): ?>
                <li class="page-item">
                    <a class="page-link" href="?page=<?php echo $page - 1; ?>">
                        <i class="fas fa-chevron-left"></i>
                    </a>
                </li>
                <?php endif; ?>

                <?php
                $startPage = max(1, $page - 2);
                $endPage = min($totalPages, $page + 2);
                
                if ($startPage > 1) {
                    echo '<li class="page-item"><a class="page-link" href="?page=1">1</a></li>';
                    if ($startPage > 2) {
                        echo '<li class="page-item disabled"><span class="page-link">...</span></li>';
                    }
                }

                for ($i = $startPage; $i <= $endPage; $i++) {
                    echo '<li class="page-item' . ($i === $page ? ' active' : '') . '">';
                    echo '<a class="page-link" href="?page=' . $i . '">' . $i . '</a>';
                    echo '</li>';
                }

                if ($endPage < $totalPages) {
                    if ($endPage < $totalPages - 1) {
                        echo '<li class="page-item disabled"><span class="page-link">...</span></li>';
                    }
                    echo '<li class="page-item"><a class="page-link" href="?page=' . $totalPages . '">' . $totalPages . '</a></li>';
                }
                ?>

                <?php if ($page < $totalPages): ?>
                <li class="page-item">
                    <a class="page-link" href="?page=<?php echo $page + 1; ?>">
                        <i class="fas fa-chevron-right"></i>
                    </a>
                </li>
                <?php endif; ?>
            </ul>
        </nav>
    </div>
    <?php endif; ?>
</div>

<!-- 删除确认模态框 -->
<div class="modal fade" id="deleteModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">确认删除</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <p>确定要删除用户 "<span id="deleteUserName"></span>" 吗？</p>
                <p class="text-danger">
                    <i class="fas fa-exclamation-triangle me-2"></i>
                    此操作将同时删除该用户的所有动态和访问记录！
                </p>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">取消</button>
                <button type="button" class="btn btn-danger" id="confirmDelete">
                    <i class="fas fa-trash-alt me-2"></i>确定删除
                </button>
            </div>
        </div>
    </div>
</div>

<!-- 用户管理脚本 -->
<script>
document.addEventListener('DOMContentLoaded', function() {
    // 删除用户
    let userIdToDelete = null;
    const deleteModal = new bootstrap.Modal(document.getElementById('deleteModal'));

    document.querySelectorAll('.delete-user').forEach(button => {
        button.addEventListener('click', function() {
            userIdToDelete = this.dataset.id;
            document.getElementById('deleteUserName').textContent = this.dataset.name;
            deleteModal.show();
        });
    });

    document.getElementById('confirmDelete').addEventListener('click', function() {
        if (userIdToDelete) {
            fetch(`/admin/api/delete_user.php?id=${userIdToDelete}`, {
                method: 'POST'
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    location.reload();
                } else {
                    alert(data.message || '删除失败');
                }
            })
            .catch(error => {
                alert('删除失败');
            });
        }
    });

    // 搜索功能
    const searchInput = document.getElementById('searchInput');
    const searchBtn = document.getElementById('searchBtn');
    const tbody = document.querySelector('tbody');

    function searchUsers() {
        const keyword = searchInput.value.toLowerCase();
        const rows = tbody.getElementsByTagName('tr');

        for (let row of rows) {
            const nickname = row.cells[1].textContent.toLowerCase();
            row.style.display = nickname.includes(keyword) ? '' : 'none';
        }
    }

    searchBtn.addEventListener('click', searchUsers);
    searchInput.addEventListener('keyup', function(e) {
        if (e.key === 'Enter') {
            searchUsers();
        }
    });
});
</script>

<?php require_once 'layout/footer.php'; ?> 