<?php
/**
 * 朋友圈访客查询系统
 * 
 * @author    锅岛主
 * @link      https://ggdao.net
 * @wechat    IJG55555
 * @copyright Copyright (c) 2026 锅岛主 (https://ggdao.net)
 * @license   保留所有权利
 */

require_once '../config/config.php';
require_once '../includes/Database.php';
require_once '../includes/Admin.php';

// 检查管理员是否登录
if (!isset($_SESSION['admin'])) {
    header('Location: /admin/login.php');
    exit;
}

$db = Database::getInstance();

// 处理VIP状态修改
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $userId = $_POST['user_id'];
    $vipLevel = $_POST['vip_level'];
    $vipExpireTime = $_POST['vip_expire_time'];
    
    $data = [
        'vip_level' => $vipLevel,
        'vip_expire_time' => $vipExpireTime ? $vipExpireTime : null
    ];
    
    $db->update('users', $data, 'id = ?', [$userId]);
    $_SESSION['success'] = '用户VIP状态更新成功！';
    
    header('Location: /admin/user_vip.php');
    exit;
}

// 获取所有用户列表
$users = $db->query("
    SELECT u.*, 
           COUNT(DISTINCT p.id) as post_count,
           COUNT(DISTINCT v.id) as visitor_count
    FROM users u
    LEFT JOIN posts p ON u.id = p.user_id
    LEFT JOIN visitors v ON u.id = v.visitor_id
    GROUP BY u.id
    ORDER BY u.created_at DESC
")->fetchAll();

// 加载管理后台模板
require_once 'layout/header.php';
?>

<div class="content-wrapper">
    <div class="content-header">
        <div class="container-fluid">
            <div class="row mb-2">
                <div class="col-sm-6">
                    <h1 class="m-0">用户VIP管理</h1>
                </div>
            </div>
        </div>
    </div>

    <section class="content">
        <div class="container-fluid">
            <?php if (isset($_SESSION['success'])): ?>
            <div class="alert alert-success alert-dismissible fade show" role="alert">
                <?php 
                echo $_SESSION['success'];
                unset($_SESSION['success']);
                ?>
                <button type="button" class="close" data-bs-dismiss="alert">
                    <span aria-hidden="true">&times;</span>
                </button>
            </div>
            <?php endif; ?>

            <div class="card">
                <div class="card-header">
                    <h3 class="card-title">用户列表</h3>
                </div>
                <div class="card-body">
                    <table class="table table-bordered">
                        <thead>
                            <tr>
                                <th>ID</th>
                                <th>头像</th>
                                <th>昵称</th>
                                <th>注册时间</th>
                                <th>动态数</th>
                                <th>访问数</th>
                                <th>VIP状态</th>
                                <th>到期时间</th>
                                <th style="width: 100px">操作</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($users as $user): ?>
                            <tr>
                                <td><?php echo $user['id']; ?></td>
                                <td>
                                    <img src="<?php echo $user['avatar']; ?>" 
                                         class="img-circle" 
                                         width="40" 
                                         height="40"
                                         alt="<?php echo htmlspecialchars($user['nickname']); ?>">
                                </td>
                                <td><?php echo htmlspecialchars($user['nickname']); ?></td>
                                <td><?php echo date('Y-m-d H:i', strtotime($user['created_at'])); ?></td>
                                <td><?php echo $user['post_count']; ?></td>
                                <td><?php echo $user['visitor_count']; ?></td>
                                <td>
                                    <span class="badge badge-<?php echo $user['vip_level'] == 1 ? 'success' : 'secondary'; ?>">
                                        <?php echo $user['vip_level'] == 1 ? 'VIP会员' : '普通用户'; ?>
                                    </span>
                                </td>
                                <td>
                                    <?php 
                                    if ($user['vip_expire_time']) {
                                        echo date('Y-m-d H:i', strtotime($user['vip_expire_time']));
                                    } else {
                                        echo '-';
                                    }
                                    ?>
                                </td>
                                <td>
                                    <button type="button" 
                                            class="btn btn-sm btn-info edit-vip"
                                            data-user='<?php echo json_encode($user); ?>'>
                                        编辑
                                    </button>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </section>
</div>

<!-- 编辑VIP状态模态框 -->
<div class="modal fade" id="vipModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <form action="" method="post">
                <input type="hidden" name="user_id" id="userId">
                <div class="modal-header">
                    <h5 class="modal-title">编辑VIP状态</h5>
                    <button type="button" class="close" data-bs-dismiss="modal">
                        <span>&times;</span>
                    </button>
                </div>
                <div class="modal-body">
                    <div class="form-group">
                        <label>用户昵称</label>
                        <input type="text" class="form-control" id="userNickname" readonly>
                    </div>
                    <div class="form-group">
                        <label>VIP状态</label>
                        <select class="form-control" name="vip_level">
                            <option value="0">普通用户</option>
                            <option value="1">VIP会员</option>
                        </select>
                    </div>
                    <div class="form-group">
                        <label>到期时间</label>
                        <input type="datetime-local" 
                               class="form-control" 
                               name="vip_expire_time">
                        <small class="form-text text-muted">留空表示立即过期</small>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">取消</button>
                    <button type="submit" class="btn btn-primary">保存</button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
// 确保在DOM和jQuery加载完成后执行
document.addEventListener('DOMContentLoaded', function() {
    // 检查jQuery是否加载
    if (typeof jQuery === 'undefined') {
        console.error('jQuery未加载，请检查footer.php中的jQuery引入');
        return;
    }
    
    // 编辑VIP状态
    document.querySelectorAll('.edit-vip').forEach(button => {
        button.addEventListener('click', function() {
            const user = JSON.parse(this.dataset.user);
            const modal = $('#vipModal');
        
        modal.find('#userId').val(user.id);
        modal.find('#userNickname').val(user.nickname);
        modal.find('select[name="vip_level"]').val(user.vip_level);
        
        if (user.vip_expire_time) {
            const expireTime = new Date(user.vip_expire_time);
            const year = expireTime.getFullYear();
            const month = String(expireTime.getMonth() + 1).padStart(2, '0');
            const day = String(expireTime.getDate()).padStart(2, '0');
            const hours = String(expireTime.getHours()).padStart(2, '0');
            const minutes = String(expireTime.getMinutes()).padStart(2, '0');
            modal.find('input[name="vip_expire_time"]').val(`${year}-${month}-${day}T${hours}:${minutes}`);
        } else {
            modal.find('input[name="vip_expire_time"]').val('');
        }
        
        modal.modal('show');
        });
    });
});
</script>

<?php require_once 'layout/footer.php'; ?> 