<?php
/**
 * 朋友圈访客查询系统
 * 
 * @author    锅岛主
 * @link      https://ggdao.net
 * @wechat    IJG55555
 * @copyright Copyright (c) 2026 锅岛主 (https://ggdao.net)
 * @license   保留所有权利
 */

require_once '../config/config.php';
require_once '../includes/Database.php';
require_once '../includes/Admin.php';
require_once '../includes/User.php';
require_once '../includes/Post.php';

$admin = new Admin();
$admin->checkAccess();

$user = new User();
$post = new Post();

// 获取日期参数
$date = isset($_GET['date']) ? $_GET['date'] : date('Y-m-d');

// 获取统计数据
$userStats = $user->getGrowthStats($date);
$postStats = $post->getShareStats($date);

// 获取最近7天的趋势数据
$trendDays = 7;
$trends = [];
for ($i = $trendDays - 1; $i >= 0; $i--) {
    $trendDate = date('Y-m-d', strtotime("-$i days"));
    $dayUserStats = $user->getGrowthStats($trendDate);
    $dayPostStats = $post->getShareStats($trendDate);
    
    $trends[] = [
        'date' => date('m-d', strtotime($trendDate)), // 只显示月-日
        'new_users' => $dayUserStats['total_new_users'],
        'share_users' => $dayUserStats['share_users'],
        'new_posts' => $dayPostStats['total_posts'],
        'effective_shares' => $dayPostStats['effective_shares']
    ];
}

$pageTitle = '增长统计';
$pageIcon = 'fas fa-chart-line';
require 'layout/header.php';
?>

<!-- 引入Chart.js -->
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>

<div class="row">
    <!-- 日期选择器 -->
    <div class="col-12 mb-4">
        <div class="card">
            <div class="card-body">
                <form method="GET" class="row g-3 align-items-center">
                    <div class="col-auto">
                        <label for="date" class="form-label">选择日期</label>
                    </div>
                    <div class="col-auto">
                        <input type="date" class="form-control" id="date" name="date" 
                               value="<?php echo htmlspecialchars($date); ?>"
                               max="<?php echo date('Y-m-d'); ?>">
                    </div>
                    <div class="col-auto">
                        <button type="submit" class="btn btn-primary">查看统计</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- 趋势图表 -->
    <div class="col-12 mb-4">
        <div class="card">
            <div class="card-header">
                <h5 class="card-title mb-0">
                    <i class="fas fa-chart-line me-2"></i>最近7天增长趋势
                </h5>
            </div>
            <div class="card-body p-4">
                <div style="height: 400px; max-width: 1200px; margin: 0 auto;">
                    <canvas id="trendChart"></canvas>
                </div>
            </div>
        </div>
    </div>

    <!-- 用户增长统计 -->
    <div class="col-md-6 mb-4">
        <div class="card h-100">
            <div class="card-header">
                <h5 class="card-title mb-0">
                    <i class="fas fa-users me-2"></i>用户增长
                </h5>
            </div>
            <div class="card-body">
                <div class="row g-3">
                    <div class="col-6">
                        <div class="border rounded p-3">
                            <div class="small text-muted">新增用户</div>
                            <div class="h3 mb-0"><?php echo number_format($userStats['total_new_users']); ?></div>
                        </div>
                    </div>
                    <div class="col-6">
                        <div class="border rounded p-3">
                            <div class="small text-muted">自然流量</div>
                            <div class="h3 mb-0"><?php echo number_format($userStats['direct_users']); ?></div>
                        </div>
                    </div>
                    <div class="col-6">
                        <div class="border rounded p-3">
                            <div class="small text-muted">裂变获取</div>
                            <div class="h3 mb-0"><?php echo number_format($userStats['share_users']); ?></div>
                        </div>
                    </div>
                    <div class="col-6">
                        <div class="border rounded p-3">
                            <div class="small text-muted">裂变用户活跃</div>
                            <div class="h3 mb-0"><?php echo number_format($userStats['share_active_users']); ?></div>
                        </div>
                    </div>
                </div>

                <?php if ($userStats['total_new_users'] > 0): ?>
                <div class="mt-4">
                    <div class="mb-2">用户来源分布</div>
                    <div class="progress" style="height: 24px;">
                        <?php
                        $sharePercent = $userStats['total_new_users'] > 0 
                            ? round(($userStats['share_users'] / $userStats['total_new_users']) * 100) 
                            : 0;
                        ?>
                        <div class="progress-bar bg-success" style="width: <?php echo $sharePercent; ?>%">
                            裂变 <?php echo $sharePercent; ?>%
                        </div>
                        <div class="progress-bar bg-primary" style="width: <?php echo (100 - $sharePercent); ?>%">
                            自然 <?php echo (100 - $sharePercent); ?>%
                        </div>
                    </div>
                </div>
                <?php endif; ?>

                <?php if ($userStats['share_users'] > 0): ?>
                <div class="mt-4">
                    <div class="mb-2">裂变用户活跃度</div>
                    <div class="progress" style="height: 24px;">
                        <?php
                        $activePercent = $userStats['share_users'] > 0 
                            ? round(($userStats['share_active_users'] / $userStats['share_users']) * 100) 
                            : 0;
                        ?>
                        <div class="progress-bar bg-success" style="width: <?php echo $activePercent; ?>%">
                            活跃 <?php echo $activePercent; ?>%
                        </div>
                        <div class="progress-bar bg-secondary" style="width: <?php echo (100 - $activePercent); ?>%">
                            未活跃 <?php echo (100 - $activePercent); ?>%
                        </div>
                    </div>
                </div>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <!-- 动态分享统计 -->
    <div class="col-md-6 mb-4">
        <div class="card h-100">
            <div class="card-header">
                <h5 class="card-title mb-0">
                    <i class="fas fa-share-alt me-2"></i>动态分享
                </h5>
            </div>
            <div class="card-body">
                <div class="row g-3">
                    <div class="col-6">
                        <div class="border rounded p-3">
                            <div class="small text-muted">新增动态</div>
                            <div class="h3 mb-0"><?php echo number_format($postStats['total_posts']); ?></div>
                        </div>
                    </div>
                    <div class="col-6">
                        <div class="border rounded p-3">
                            <div class="small text-muted">分享次数</div>
                            <div class="h3 mb-0"><?php echo number_format($postStats['shared_posts']); ?></div>
                        </div>
                    </div>
                    <div class="col-12">
                        <div class="border rounded p-3">
                            <div class="small text-muted">有效分享（有访问）</div>
                            <div class="h3 mb-0"><?php echo number_format($postStats['effective_shares']); ?></div>
                        </div>
                    </div>
                </div>

                <?php if ($postStats['shared_posts'] > 0): ?>
                <div class="mt-4">
                    <div class="mb-2">分享转化率</div>
                    <div class="progress" style="height: 24px;">
                        <?php
                        $effectivePercent = $postStats['shared_posts'] > 0 
                            ? round(($postStats['effective_shares'] / $postStats['shared_posts']) * 100) 
                            : 0;
                        ?>
                        <div class="progress-bar bg-success" style="width: <?php echo $effectivePercent; ?>%">
                            有效 <?php echo $effectivePercent; ?>%
                        </div>
                        <div class="progress-bar bg-secondary" style="width: <?php echo (100 - $effectivePercent); ?>%">
                            未访问 <?php echo (100 - $effectivePercent); ?>%
                        </div>
                    </div>
                </div>
                <?php endif; ?>

                <?php if ($postStats['total_posts'] > 0): ?>
                <div class="mt-4">
                    <div class="mb-2">动态分享率</div>
                    <div class="progress" style="height: 24px;">
                        <?php
                        $sharePercent = $postStats['total_posts'] > 0 
                            ? round(($postStats['shared_posts'] / $postStats['total_posts']) * 100) 
                            : 0;
                        ?>
                        <div class="progress-bar bg-primary" style="width: <?php echo $sharePercent; ?>%">
                            已分享 <?php echo $sharePercent; ?>%
                        </div>
                        <div class="progress-bar bg-secondary" style="width: <?php echo (100 - $sharePercent); ?>%">
                            未分享 <?php echo (100 - $sharePercent); ?>%
                        </div>
                    </div>
                </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<script>
document.getElementById('date').addEventListener('change', function() {
    this.form.submit();
});

// 准备趋势图数据
const trends = <?php echo json_encode($trends); ?>;
const dates = trends.map(t => t.date);
const newUsers = trends.map(t => t.new_users);
const shareUsers = trends.map(t => t.share_users);
const newPosts = trends.map(t => t.new_posts);
const effectiveShares = trends.map(t => t.effective_shares);

// 绘制趋势图
const ctx = document.getElementById('trendChart').getContext('2d');
new Chart(ctx, {
    type: 'line',
    data: {
        labels: dates,
        datasets: [
            {
                label: '新增用户',
                data: newUsers,
                borderColor: 'rgb(75, 192, 192)',
                backgroundColor: 'rgba(75, 192, 192, 0.1)',
                tension: 0.1,
                fill: true,
                pointRadius: 4,
                borderWidth: 2
            },
            {
                label: '裂变用户',
                data: shareUsers,
                borderColor: 'rgb(255, 99, 132)',
                backgroundColor: 'rgba(255, 99, 132, 0.1)',
                tension: 0.1,
                fill: true,
                pointRadius: 4,
                borderWidth: 2
            },
            {
                label: '新增动态',
                data: newPosts,
                borderColor: 'rgb(54, 162, 235)',
                backgroundColor: 'rgba(54, 162, 235, 0.1)',
                tension: 0.1,
                fill: true,
                pointRadius: 4,
                borderWidth: 2
            },
            {
                label: '有效分享',
                data: effectiveShares,
                borderColor: 'rgb(153, 102, 255)',
                backgroundColor: 'rgba(153, 102, 255, 0.1)',
                tension: 0.1,
                fill: true,
                pointRadius: 4,
                borderWidth: 2
            }
        ]
    },
    options: {
        responsive: true,
        maintainAspectRatio: true,
        aspectRatio: 3,
        interaction: {
            mode: 'index',
            intersect: false,
        },
        plugins: {
            legend: {
                position: 'top',
                align: 'start',
                labels: {
                    boxWidth: 15,
                    usePointStyle: true,
                    padding: 20,
                    font: {
                        size: 13
                    }
                }
            },
            tooltip: {
                mode: 'index',
                intersect: false,
                padding: 10,
                backgroundColor: 'rgba(0, 0, 0, 0.8)',
                titleColor: '#fff',
                bodyColor: '#fff',
                borderColor: 'rgba(255, 255, 255, 0.1)',
                borderWidth: 1,
                bodyFont: {
                    size: 13
                },
                titleFont: {
                    size: 13
                }
            }
        },
        scales: {
            y: {
                beginAtZero: true,
                grid: {
                    color: 'rgba(0, 0, 0, 0.05)'
                },
                ticks: {
                    font: {
                        size: 12
                    }
                }
            },
            x: {
                grid: {
                    display: false
                },
                ticks: {
                    font: {
                        size: 12
                    }
                }
            }
        }
    }
});
</script>

<?php require 'layout/footer.php'; ?> 